//============================================================================
// Weather Shield for Weather Station Data Logger  
// 
//
//This application is free software; you can redistribute it and/or
//modify it under the terms of the GNU Lesser General Public
//License as published by the Free Software Foundation; either
//version 3 of the License, or (at your option) any later version.
//
//This application is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR When PARTICULAR PURPOSE. See the GNU
//Lesser General Public License for more details.
//
//You should have received a copy of the GNU Lesser General Public
//License along with this library; if not, see <http://www.gnu.org/licenses/>
//
//=============================================================================

#include "Arduino.h"
// #include <wiring.h>
#include <avr/interrupt.h>

// this header file defines the reciever hardware to be supported
#include "WxReceiverConfig.h"

//
// to generate a version of the sketch which does not illuminate the 
// LED lamps, set this to zero to activate the empty macros.
//
#define ENABLE_LEDS					1
#define ENABLE_HDWR_DEBUG_FLAGS		0

#include "wxrx.h"

//
// macros to control and test the trigger polarity for timer 1B
//

#define INPUT_CAPTURE_IS_RISING_EDGE()    ((TCCR1B & _BV(ICES1)) != 0)
#define INPUT_CAPTURE_IS_FALLING_EDGE()   ((TCCR1B & _BV(ICES1)) == 0)
#define SET_INPUT_CAPTURE_RISING_EDGE()   (TCCR1B |=  _BV(ICES1))
#define SET_INPUT_CAPTURE_FALLING_EDGE()  (TCCR1B &= ~_BV(ICES1))
//
// macros to control the two LEDs on the weather shields
//
#if ENABLE_LEDS

#if WX_SHIELD_VERSION == 2

#define RX_LED_ON            PORTD |= _BV(RECEIVING_LED_PIN)
#define RX_LED_OFF           PORTD &= (byte)(~(_BV(RECEIVING_LED_PIN)))

#define LOCK_LED_ON          PORTD |= _BV(LOCKED_LED_PIN)
#define LOCK_LED_OFF         PORTD &= (byte)(~(_BV(LOCKED_LED_PIN)))
#define LOCK_LED_STATE	     (PORTD & _BV(LOCKED_LED_PIN))

#endif

#if WX_SHIELD_VERSION == 1

#define RX_LED_ON		digitalWrite(RECEIVING_LED_PIN, HIGH)
#define RX_LED_OFF		digitalWrite(RECEIVING_LED_PIN, LOW)

#define LOCK_LED_ON		digitalWrite(LOCKED_LED_PIN, HIGH)
#define LOCK_LED_OFF	digitalWrite(LOCKED_LED_PIN, LOW)
#define LOCK_LED_STATE  digitalRead(LOCKED_LED_PIN)

#endif

#else

#define RX_LED_ON 
#define RX_LED_OFF 
#define LOCK_LED_ON 
#define LOCK_LED_OFF 
#define LOCK_LED_STATE false

#endif
//
// some macros to make some of the code below a little more obvious
//
// macros to reset the receive state machine state. The unsafe version
// can only be used in an interrupt. 
//
#define UNSAFE_WEATHER_RESET() { protocol_version = short_count = long_count = 0; rx_state = RX_STATE_IDLE; RX_LED_OFF; want_agc_frozen = false; }  
#define WEATHER_RESET() { noInterrupts(); protocol_version = short_count = long_count = 0; rx_state = RX_STATE_IDLE; RX_LED_OFF; want_agc_frozen = false; interrupts(); }  
//
// values for timer 2's control registers
//
#define PULSE_TIMEOUT_ENABLE  _BV(TOIE2)
#define PULSE_TIMEOUT_DISABLE 0
#define T2_PRESCALE_X128 ( _BV(CS22) | _BV(CS20) )
#define T2_PRESCALE_X256 ( _BV(CS22) | _BV(CS21) )
#define T2_PRESCALE_X1024 ( _BV(CS22) | _BV(CS21) | _BV(CS20) )
#define PULSE_TIMEOUT_FLAG _BV(TOV2)
//
// Preamble information
//
// this defines the minimum number of RF transitions required to identify
// a preamble sequence. Perfect version 1.0, 2.1 and 3.0 preambles contain
// 24, 32 and 24 transistions respectively.
//
// The maximum number determines when a string of pulses is not considered
// a valid preamble if they just keep going on.
//
#define V3_MIN_SYNC_COUNT           16
#define V3_MAX_SYNC_COUNT           48
#define V2_MIN_SYNC_COUNT           24
#define V2_MAX_SYNC_COUNT			64
#define V1_MIN_SYNC_COUNT			16
#define V1_MAX_SYNC_COUNT			48
//
// State machine states
//
#define RX_STATE_IDLE               0  /* ready to go */
#define RX_STATE_RECEIVING_V3       1  /* receiving version 3 protocol messsage */
#define RX_STATE_PACKET_RECEIVED    2  /* complete message received */
#define RX_STATE_RECEIVING_V2       3  /* receiving version 2 protocol message */
#if ACCEPT_SQUARE_WAVE
#define RX_STATE_RECEIVING_SQ       4
#endif
#define RX_STATE_V1_PREAMBLE		10
#define RX_STATE_V1_SYNC_PULSE		11
#define RX_STATE_V1_SYNC_END			12
#define RX_STATE_RECEIVING_V1		13
#if DETECT_PSM
#define RX_STATE_AR_PREAMBLE		14
#define RX_STATE_RECEIVING_AR		15
#endif

#define BIT_ZERO                    0
#define BIT_ONE                     1



// *** RF Protocol Decoding Variables ***

//
// timer value at last edge capture event. used to compute time interval
// between previous event and current event.
//
static unsigned int timer1_ovfl_count;
static unsigned int previous_captured_time;
static unsigned int v1_sync_end;
//
// variables used to time the illumination of the red LED.
//
static unsigned int lock_led_on_time;
//
// receive buffer pointer 
// using bit fields like this is platform dependent and generally not recommended.
// however, since this code will never (never say never :-O ) run on anything but an Atmel processor,
// the risks are minimized. on the plus side, this union with bit fields makes the code
// cleaner and easier to understand.
//
static union {
	unsigned int value;  
	struct 
	{
		unsigned int bit:2;      // two LSBs refer to a bit within the nibble
		unsigned int nibble:14;  // 14 MSBs refer to a nibble 
	} portion;
} bufptr;
//
// receive buffer
// only the lower nibble of each byte is used. The upper nibbles
// are zeroed in the reset() function and should not be modified
// thereafter.
//
static byte packet[MAX_MSG_LEN];
//
// while looking for a valid preamble, these track the number of 
// short and long periods seen. for now, preambles are much less
// than 256 pulses long, but use 16-bit counters just to be safe
//
static unsigned int short_count;
static unsigned int long_count;
//
// we keep track of time in units of 1/2 clock period
// when this value is even we are at the middle of a clock period.
//
static int half_clock_count;
//
// this should only ever be set to 1, 2 or 3
//
static byte protocol_version; // protocol version of the current message
//
// for the version 2 protocol, this is used as a toggle to cause every
// other bit to be "dumped"-- bits come in inverted pairs and the first
// bit in each pair is inverted, so we dump the first bit and keep the 2nd.
//
static boolean dump_bit;

static byte rx_state; // current state of the receive state machine

//
// for this to work properly, the main run loop must call the WxRx::step()
// method periodically. This will update the receiver's AGC state according
// to the value of this variable. Also note, that only the Version 2 WxShield
// offers control of AGC state. Earlier shields cannot use this capability.
//
static boolean want_agc_frozen;

static byte adc_sra;

const unsigned long mm_diff = 0x7fffffffUL; 

//
// pulse timing limits for protocol versions 2.1 and 3.0
//
// thresholds for short and long timer periods. the thresholds are different
// when the RF signal is on and off. the three values in each array are, in order:
// minimum short period, threshold between short and long periods, and maximum long period.
// if the timer value is exactly equal to the short/long threshold it does not matter 
// which choice is made -- this is probably a noise reading anyway. 
// periods less then the minimum short or maximum long periods are considered noise and
// rejected.
//
const unsigned int rf_off_thresholds[3] = { 100, 212, 350 };  // for a 4usec timer tick, 400,848,1400 usec
const unsigned int rf_on_thresholds[3]  = {  50, 154, 275 };  // for a 4usec timer tick, 200,616,1100 usec
//
// timing limits for protocol version 1.0
// lucky for us, the minimum on period in the preamble (1500usec) is longer than the maximum
// on period for both version 2.1 and 3.0 protocols which allows us to detect version 1.0 messages instantly
// in the first on pulse.
//
const unsigned int v1_off_thresholds[3] = { 243, 488, 725 }; //  970,1950,2900 usec
const unsigned int v1_on_thresholds[3]  = { 375, 600, 850 }; // 1500,2400,3400 usec
const unsigned int v1_sync_off[2] = { 1000, 1100 };          // 4000,4400 usec
const unsigned int v1_sync_on[2]  = { 1396, 1496 };          // 5585,5985 usec
const unsigned int v1_sync_end_short[2] = { 1250, 1350 };    // 5000,5400 usec
const unsigned int v1_sync_end_long[2]  = { 1625, 1725 };	 // 6500,6900 usec

#ifdef DETECT_PSM
const unsigned int acr_on_thresholds[3]   = { 75, 200, 201 };		// 400, 600 usec for the on pulses
const unsigned int acr_sync_off[2]		  = { 2125, 2500 };			// 8.5, 10 msec
const unsigned int acr_off_thresholds[3]  = { 425, 750, 1150 };		// 1700, 3000, 4600 usec
#endif

#if WX_SHIELD_VERSION == 2
//
// flag used to trigger capture of RSSI value during first transition after the preamble,
// and the RSSI value measured during that process.
//
static byte capture_rssi;

static unsigned int rssi; // analog rssi value read from the ADC in the Atmel processor.

#endif

//
// Overflow interrupt routine for timer 2
// When the last bit of a message has been received by the event capture ISR, the state machine will just 
// sit there waiting for the next RF transition -- which won't occur until a new message begins,
// or the receiver's AGC adjusts to the lack of signal and starts detecting noise.
// This timer is designed to expire shortly after the last message bit, so the new message
// can be signalled to background software. This will then happen much sooner than if we waited
// for a new message or noise to begin arriving.
//
ISR(TIMER2_OVF_vect)
{
	byte mask = TIMSK2;
	TIMSK2 =  PULSE_TIMEOUT_DISABLE; // disable further interrupts
	//
	// just for safety, make sure that timer 2 overflow interrupts are enabled before proceeding
	//
	if (!(mask & PULSE_TIMEOUT_ENABLE)) return;

	boolean rx_state_v1 = (rx_state == RX_STATE_RECEIVING_V1);

	boolean rx_state_v23 = 
		(rx_state == RX_STATE_RECEIVING_V2) || 
		(rx_state == RX_STATE_RECEIVING_V3);

	byte min_cnt = (rx_state_v1 ? 24 : 40);

#if DETECT_PSM
	boolean rx_state_ar = rx_state == RX_STATE_RECEIVING_AR;
	if (rx_state_ar) min_cnt = 24;


	if ((rx_state_v1 || rx_state_v23 || rx_state_ar) && (bufptr.value > min_cnt))
#else
	if ((rx_state_v1 || rx_state_v23) && (bufptr.value > min_cnt))
#endif
	{
		rx_state = RX_STATE_PACKET_RECEIVED;
	}
	else
	{
		UNSAFE_WEATHER_RESET();
	}	
}
//
// Overflow interrupt vector
// this is need to keep track of overflow events on timer 1 which allows detection
// of very long time periods  while looking for preambles when there might be a lot
// of time between data transitions (or not, depending on how much noise is being detected).
//
ISR(TIMER1_OVF_vect)
{
	timer1_ovfl_count++;
}

void led_update_check()
{
	// most of the time, the LED will be off and we'll return immediately.
	//
	if (LOCK_LED_STATE)
	{
		noInterrupts();
		unsigned int time_expired;
		unsigned int current_time = ICR1;
		//
		// computing the period with an int (16 bits), we can only go about 1/4 of a second before 
		// overflowing. however, the client is expected to call us much more often than that
		// from the run loop. worst thing that happens if not is that the led will stay on 
		// for a longer time period.
		//
		if (current_time < lock_led_on_time)
		{
			// do the math using signed 32-bit integers, but convert the result back to a 16-bit integer
			time_expired = (unsigned int)((long)current_time + 0x10000L - (long)lock_led_on_time);    
		}
		else
		{
			time_expired = current_time - lock_led_on_time;
		}

		if (time_expired > 25000U) // 100msec on time
		{
			LOCK_LED_OFF;
		}
		interrupts();
	}
}

//
// Event capture interrupt routine for timer 1
//
// This is fired every time the received RF signal goes on or off. Since every 
// transition (up or down) is followed by an opposite transition (down or up), the edge detection
// bit for the timer is flipped with every transition.
//
// Decoding of the Manchester-coded ASK is performed in this ISR. The technique works by
// examining the time between adjacent RF transitions. This is easy to do using the Atmel
// processor's "timer 1" which has an input specifically designed to detect the timing of transitions
// on input bit 0 of port B. This input even has a de-glitching feature which helps to filter
// out noise on the RF transitions.
//
// It may take a little thinking, but you can prove to yourself these facts about normal
// Manchester-coded signals:
// 
// 1) If the next message bit is the same as the previous message bit, there will need to 
//    be an RF transition (on-to-off or vice versa) in the middle of the bit period.
// 2) If the next message bit is the opposite of the previous bit, there will be no RF transition
//    in the middle of the bit period.
// 3) The time between transitions is either short (1/2 bit period) or long (a full bit period).
// 4) When short transitions occur, they always occur in pairs.
// 
// The ISR below uses these ideas to decode the signal, with the additional knowledge that all 
// bits in the preamble of short transitions are "1" bits. When a long transition occurs, it 
// represents a bit that is the opposite of the preceeding bit. When two short transitions occur,
// it represents a bit that is identical to the preceeding bit.
//
// Special considerations are required for OS protocol version 2.1 messages. 
// See the RF protocol definition document for details.
//
// In fact, RF transmissions from OS units do not have a 50% duty cycle so the definition of
// short and long bit periods are a bit skewed (see protocol document).
//
ISR(TIMER1_CAPT_vect)
{ 
	// do the time-sensitive things first
	TIMSK2 = PULSE_TIMEOUT_DISABLE; // disable timeout events from timer 2
	TCNT2 = 64U; // reset the counter
	// TCNT2 =  ((rx_state == RX_STATE_RECEIVING_V2) || (rx_state == RX_STATE_RECEIVING_V3)) ? 64U : 0U;

	unsigned int ovfl = timer1_ovfl_count;
	timer1_ovfl_count = 0;
	//
	// grab the event time
	//
	unsigned int captured_time = ICR1;
	//
	// depending on which edge (rising/falling) caused this interrupt, setup to receive the opposite
	// edge (falling/rising) as the next event.
	//
	boolean rf_was_on = INPUT_CAPTURE_IS_FALLING_EDGE();

	if(!rf_was_on)
		SET_INPUT_CAPTURE_FALLING_EDGE();
	else 
		SET_INPUT_CAPTURE_RISING_EDGE();
	//
	// detect and deal with timer overflows. the timer will overflow about once every
	// 0.26 seconds so it is not all that rare of an occurance. as long as there has only been 
	// one overflow this will work. If there is more than one overflow, just set the period to maximum.
	// there IS a race condition where this logic will sometimes fail but the window is very small.
	// here's the situation: an input edge happens about the same time as timer1 overflows,
	// and the timer gets latched before the overflow, but the timer overflow ISR runs before the
	// edge capture ISR (don't know if this is really possible). In this case, the timer will not
	// have overflowed (it will be equal to 0xFFFF) but the overflow counter WILL indicate an 
	// overflow. Again, the window to create this problem is so narrow this should rarely happen.
	// the worst thing that will happen is that we'll occasionally loose a preamble bit or maybe
	// even a whole message (even more rare). It's certainly better than ignoring overflows altogether.
	//
	// the other thing that can happen if there is no activity on the DATA line for 4.7 hours, is that
	// the overflow counter itself will overflow. again, this might cause us to miss one message at the 
	// worst case, so this is an acceptable risk. In real life, there is always noise on the DATA line
	// if no signal is present which makes this event even more unlikely.
	//
	unsigned int captured_period;
	unsigned int v1_sync_end_period;

	if (ovfl > 1)
	{
		captured_period = 0xFFFFL;
	}
	else
	{
		//
		// computing the period with an int (16 bits) can only go about 1/4 of a second before 
		// overflowing. however, the timeout provided by timer 2 ensures that we'll never see
		// a period that long. the overflow ISR will force the rx state back to IDLE, which does
		// not care about the result of this calculation.
		//
		if (captured_time < previous_captured_time)
		{
			// do the math using signed 32-bit integers, but convert the result back to a 16-bit integer
			captured_period = (unsigned int)((long)captured_time + 0x10000L - (long)previous_captured_time);    
		}
		else
		{
			captured_period = captured_time - previous_captured_time;
		}
	}

	unsigned int *thresholds;
	if (rx_state == RX_STATE_RECEIVING_V1) 
	{
		thresholds = rf_was_on ? v1_on_thresholds : v1_off_thresholds;
	}
#if DETECT_PSM
	else if (rx_state == RX_STATE_RECEIVING_AR || rx_state == RX_STATE_AR_PREAMBLE)
	{
		thresholds = rf_was_on ? acr_on_thresholds : acr_off_thresholds;
	}
#endif
	else
	{
		thresholds = rf_was_on ? rf_on_thresholds : rf_off_thresholds;
	}

	boolean short_period = false;
	boolean long_period = false;
	boolean v1_preamble_detected = false;
#if DETECT_PSM
	boolean acurite_detected = false;
#endif
	boolean v1_period_okay = false;
	boolean acurite_msg_bad = false;
	byte adc_high, adc_low;
	//
	// determine if the time between the last transition and this one represents 
	// a "short" period or a "long" period.
	//
	if (captured_period < thresholds[0])
	{
		// this is noise 
	}
	else 
	{
		if (captured_period <= thresholds[1])
		{
			short_period = true;
		}
		else
		{
			if (captured_period <= thresholds[2])
			{
				long_period = true;
			}	
			else
			{
				//
				// this interval was longer than longest threshold value. it is either invalid,
				// or perhaps releated to an OS V1.0 or Acurite signal.
				//
				v1_preamble_detected = rf_was_on && 
					(captured_period >= v1_on_thresholds[0]) && 
					(captured_period <= v1_on_thresholds[1]);
#if DETECT_PSM
				if (!v1_preamble_detected && !rf_was_on)
				{
					acurite_detected = ((captured_period >= acr_off_thresholds[0]) && 
										(captured_period <= acr_off_thresholds[2])     );
				}
#endif
			}
		}
	}

	//
	//===============================//
	//
	//    HUMONGOUS CASE STATEMENT
	//    This implements the receive
	//    state machine.
	//
	//===============================//
	//
	switch (rx_state)
	{
	//
	//===============================//
	//
	//           IDLE STATE
	//
	//===============================//
	//
	case RX_STATE_IDLE:
		//
		// Version 3 Protocol:
		// When idle, we're looking for a minimum number of short transitions which will 
		// signify the beginning of a message. It is possible the receiver will miss a few
		// of the initial pulses, so we don't want to require all of them to be received.
		// After receiving the required minimum, the first long pulse received will kick
		// us into the V3 receiving state. this is also the first bit of the sync nibble,
		// and is always a zero bit.
		//
		// Version 2.1 protocol:
		// The preamble here is one short pulse (which we often miss) followed by 16 long
		// pulses. The first short pulse received after a minimum set of long ones will
		// kick us into the V2 receiving state.
		//
		// Version 1.0 protocol:
		// the preamble is twelve short pulses and it will be easier to track it in a separate
		// state to avoid making the logic below too complex
		//
		// Acurite temperature-only sensors:
		// All pulses are about 500usec long. The packet starts with a pulse followed by a
		// long off period (acr_sync_off). The packet is repreated 8-10 times, so we won't
		// try to catch this long off period as it would require freezing the AGC at a point in 
		// time where we're not sure what's going on. Bad idea. Instead, we'll look for one of
		// the short off periods following a pulse. That tells us we may be seeing an Acurite signal,
		// and we will freeze AGC at the end of the current pulse. Then we'll just wait for the
		// long sync off period to get in sync (since the signal repeats so many times we're not
		// giving up too much this way). While AGC is frozen, we'll require on and off pulses to be 
		// in the expected range, and count a maximum of 28 valid off periods before giving up on
		// finding the longer sync off period. 
		//
		// If we do see the long sync off period, then the next off period begins the packet and
		// we can start receiving.
		//
		// the 00782W3 wireless sensor sends a series of three short pulses separated by only
		// 500usec or so prior to the long packet...perhaps sort of a wake-up call. we ignore these
		// in the current decoding logic -- they will seem like an aborted version 3 preamble.
		//
		if (v1_preamble_detected)
		{
			rx_state = RX_STATE_V1_PREAMBLE;
			short_count++;
			break;
		}
#if DETECT_PSM
		if (acurite_detected && (short_count < 10)) // ???? WTF ????
		{
			//digitalWrite(DEBUG_FLAG_1_PIN, HIGH);
			want_agc_frozen = false;  // redundant?
			rx_state = RX_STATE_AR_PREAMBLE;
			break;
		}
#endif
		if (short_period)
		{
			if ((short_count <= 1) && (long_count > V2_MIN_SYNC_COUNT))
			{
				//
				// this is the end of a valid version 2.1 message preamble.
				//
				protocol_version = 2;
				RX_LED_ON;
				LOCK_LED_OFF;
				rx_state = RX_STATE_RECEIVING_V2;
				half_clock_count = 1;
				// 
				// record the first bit
				//
				bufptr.value = 0;
				packet[0] = rf_was_on ? 1 : 0;
				//
				// read ADCH to clear the ADC registers because the timer event
				// will be the end of a short RF pulse and we'll use that to measure RSSI
				//
#if WX_SHIELD_VERSION == 2
				adc_high = ADCH;
				capture_rssi = 2;
#endif
			}
			else if (long_count == 0)
			{
				short_count++;
#if ACCEPT_SQUARE_WAVE
				if (short_count > SQ_HDR_CNT)
				{
					RX_LED_ON;
					LOCK_LED_OFF;
					protocol_version = 3;
					rx_state = RX_STATE_RECEIVING_SQ;
					bufptr.value = 1;
					packet[0] = rf_was_on ? 1 : 0;
					half_clock_count = 1;
#if WX_SHIELD_VERSION == 2
					adc_high = ADCH;
					capture_rssi = 2;
#endif
				}				
#endif
			}
			else
			{
				UNSAFE_WEATHER_RESET();
			}
		}
		else if (long_period)
		{ 
			if(short_count > V3_MIN_SYNC_COUNT) 
			{
				//
				// this is the end of a valid version 3.0 message preamble.
				//
				RX_LED_ON;
				LOCK_LED_OFF;
				rx_state = RX_STATE_RECEIVING_V3;
				protocol_version = 3;
				half_clock_count = 0; // because long_period is true
				//
				// record the first bit
				//
				bufptr.value = 1;
				packet[0] = rf_was_on ? 1 : 0;
				//
				// read ADCH to clear the ADC registers because the timer event
				// will be the end of a long RF pulse and we'll use that to measure RSSI
				//
#if WX_SHIELD_VERSION == 2
				adc_high = ADCH;
				capture_rssi = 2;
#endif
			} 
			else if (short_count <= 1)
			{
				long_count++;
			}
			else 
			{
				UNSAFE_WEATHER_RESET();
			}
		} 
		else 
		{
			UNSAFE_WEATHER_RESET();
		}
		//
		// if there is an extended sequence of either short or long periods, something
		// may be wrong, so reset the receiver state machine in an attempt to recover.
		// note: this code will not be executed if square waves are accepted because we
		// will transition to another state before the count gets that large.
		//
		if ((short_count > V3_MAX_SYNC_COUNT) || (long_count > V2_MAX_SYNC_COUNT))
		{
			UNSAFE_WEATHER_RESET();
		}

		break;
#if DETECT_PSM
	//
	//===============================//
	//
	//       ACURITE PREAMBLE STATE
	//
	//===============================//
	//
	case RX_STATE_AR_PREAMBLE:
		//
		// If we suspect an AcuRite message, there are very long off periods 
		// (between 2 and 9 msec) and we must freeze the AGC to properly receive the message.
		// This preamble state kind of a misnomer. What is happening is that we're in the middle
		// of one of the repeated messages, waiting for the end, which is signified by a 
		// very long off period (about 9 msec). To get the AGC adjusted properly we'll turn
		// it on during every single RF pulse here, and back off again after the pulse completes.
		// It takes less than 100usec to turn it on/off so this should work okay.
		//
		// Once we see a long off period, it is the end of one message repetition and we can 
		// transition to the receive state. During receive, we'll only turn the AGC on for one
		// RF pulse in each message repetition...is that enough?
		//
		// In summary, for this state:
		//
		// 1. freeze agc at the end of each "ON" period (when short_count < 3)
		// 2. terminate if anything other than short on pulse is found
		// 3. look for short/long off periods, OR the long sync off period to kick
		//    us into the receiving state
		// 4. terminate if short count becomes > 28. That's because there's at most 
		//    24 short on periods per message.
		//    If we've received that much stuff and still no long sync off period, then
		//    this cannot be an AcuRite message.
		//
		if (short_count > 28 || (rf_was_on && !short_period))
		{
			// digitalWrite(DEBUG_FLAG_1_PIN, LOW);
			want_agc_frozen = false;
			UNSAFE_WEATHER_RESET();
		}
		else
		{
			// check for invalid off period
			if (rf_was_on)
			{
				if (short_period)
				{
					// for AcuRite messages, we use short_count to count the bits in a message.
					short_count++;
					want_agc_frozen = true;
				}
				else
				{
					// digitalWrite(DEBUG_FLAG_1_PIN, LOW);
					UNSAFE_WEATHER_RESET();
				}
			}
			else
			{
				want_agc_frozen = false;

				if (!long_period && !short_period)
				{
					if (captured_period < acr_sync_off[0]  || captured_period > acr_sync_off[1])
					{
						// digitalWrite(DEBUG_FLAG_1_PIN, LOW);
						// invalid off interval detected here
						UNSAFE_WEATHER_RESET();
					}
					else
					{
						// valid sync off period detected here; start receiving packet data now
						RX_LED_ON;
						LOCK_LED_OFF;
						//
						// if we stuff the first two bits of the message into the 
						// 2 LSBs of the first nibble, then all of the data bits
						// will be properly aligned remainder of the buffer.
						// this will require that we fill in two zeros to buffer
						// when a long period is detected to create proper alignment for 
						// subsequent repetitions of the message.
						//
						packet[0] = 0;
						bufptr.value = 2; // ready to receive first bit...
						//want_agc_frozen = false;
						protocol_version = 16; // flags this as an AcuRite message
#if WX_SHIELD_VERSION == 2
						adc_high = ADCH;
						capture_rssi = 2;
#endif
						// digitalWrite(DEBUG_FLAG_1_PIN, LOW);
						rx_state = RX_STATE_RECEIVING_AR;
					}
				}	
			}
		}
		break;
	//
	//===============================//
	//
	//       ACURITE RECEIVING STATE
	//
	//===============================//
	//
	case RX_STATE_RECEIVING_AR:
		//
		// the message we get here can be repeated up to 9 or so times.
		// so what to do with all those repetitions??? if the msg buffer
		// length was increased to say 240 or so, we could save them all...
		// and do sort of a majority vote at the end as a kind of checksum...?
		// 
		// decode is simple...we get a '0' bit at the end of a short off period,
		// and a '1' bit at the end of a long off period.
		//

#if WX_SHIELD_VERSION == 2
		//
		// read the ADC to get RSSI value if necessary
		//
		if (capture_rssi > 1)
		{
			if (!rf_was_on)
			{
				// this is the beginning of an RF pulse, so enable RSSIC
				// by changing PORTC bit 1 to an input.
				DDRC &= ~0x02;
				capture_rssi = 1;
			}
		}
		else if (capture_rssi)
		{
			if (rf_was_on)
			{
				// this is the end of the RF pulse we used to measure RSSI.
				// disable RSSIC by changing PORTC bit 1 to an output (which has
				// already been programmed to the LOW state).
				//
				DDRC |= 0x02;
				//
				// start an ADC at the end of the rf pulse. if we try to start it at the beginning,
				// the RSSIOUT voltage will not yet reflect the pulse amplitude so we wait until the
				// end. 
				//
				adc_sra = ADCSRA & ~0x10; // get current value and turn off the interrupt status bit
				ADCSRA = adc_sra | 0x10;  // clear the conversion complete bit
				ADCSRA = adc_sra | 0x40;  // write a 1 to the start conversion bit
				ADCSRA = adc_sra;         // set the start bit back to zero (is this necessary?)
			}
			else
			{
				adc_sra = ADCSRA;
				if (adc_sra & 0x10)
				{
					adc_low = ADCL;
					adc_high = ADCH;
					rssi = (adc_high << 8) | adc_low;
				}
				else
				{
					// the conversion did not finish so set RSSI to zero to flag the problem
					//
					adc_low = 0;
					adc_high = 0;
					rssi = 0;
				}
				capture_rssi = 0;
			}
		}
#endif

		acurite_msg_bad = false;
		want_agc_frozen = true;

		if (rf_was_on)
		{
				if (short_period)
				{
					//want_agc_frozen = true;
				}
				else
				{
					acurite_msg_bad = true;
				}
		}
		else
		{
			// three valid cases: short, long or sync-off periods
			if (short_period || long_period)
			{				
				// detect a bit here.
				if(long_period)
				{
					packet[bufptr.portion.nibble] |= 1 << (3 - bufptr.portion.bit);
					//RX_LED_ON;
				}
				else 
				{
					packet[bufptr.portion.nibble] &= ~(1 << (3 -bufptr.portion.bit));
					//RX_LED_OFF;
				}

				bufptr.value++;
				if (bufptr.portion.nibble >= (MAX_MSG_LEN - 1))
				{
					acurite_msg_bad = true;
				}
			}
			else
			{
				// check for a sync off period
				if (captured_period >= acr_sync_off[0] && captured_period <= acr_sync_off[1])
				{
					// fill the current nibble in the output buffer with zeros
					// if all went well, it should require two zeros to be stuffed in there
					//
					while (bufptr.portion.bit)
					{
						packet[bufptr.portion.nibble] &= ~(1 << (3 -bufptr.portion.bit));
						bufptr.value++;
					}
					//
					// init the next nibble  to zero, and skip over the first two bits
					// this will properly align the next copy of the temperature message,
					// same as the previous one.
					//
					packet[bufptr.portion.nibble] = 0;
					bufptr.value += 2;
					want_agc_frozen = false; // reset agc once every repeated message ???
				}
				else
				{
					acurite_msg_bad = true;
				}
			}
			//if (!acurite_msg_bad) want_agc_frozen = false;
		}

		if (acurite_msg_bad)
		{
			if (bufptr.value > 23)
			{
				rx_state = RX_STATE_PACKET_RECEIVED;				
				RX_LED_OFF;
			}
			else
			{
				RX_LED_OFF;
				LOCK_LED_ON;
				lock_led_on_time = captured_time;
				UNSAFE_WEATHER_RESET();
			}
		}
		break;
#endif
	//
	//===============================//
	//
	//       V1 PREAMBLE STATE
	//
	//===============================//
	//
	case RX_STATE_V1_PREAMBLE:
		//
		// we're looking for the following values of captured_period here:
		// 1. A short off period
		// 2. A short on period
		// 3. The first sync-off interval
		//
		// anything else means a reset
		//
		thresholds = rf_was_on ? v1_on_thresholds : v1_off_thresholds;

		if (captured_period >= thresholds[0])
		{
			if (captured_period <= thresholds[1])
			{
				short_count++;
				v1_period_okay = true;
				if (short_count >= V1_MIN_SYNC_COUNT && rf_was_on) 
				{
					//
					// a majority of the preamble has been successfully received,
					// so this is a good time to freeze the AGC. This will make sure
					// it is frozen for both long rf-off periods that are part of the sync 
					// portion of version 1.0 RF messages.
					//
					want_agc_frozen = true;
				}
			}
			else
			{
				if (!rf_was_on && (captured_period >= v1_sync_off[0]) && (captured_period <= v1_sync_off[1]))
				{
					if (short_count > V1_MIN_SYNC_COUNT)
					{
						// 
						// this is the end of the long off period following the preamble, 
						// just prior to the long sync pulse.
						//
						v1_period_okay = true;
						rx_state = RX_STATE_V1_SYNC_PULSE;						
					}
				}
			}
		}
		if (!v1_period_okay || (short_count > V1_MAX_SYNC_COUNT)) 
		{
			UNSAFE_WEATHER_RESET();
		}
		break;
	//
	//===============================//
	//
	//         V1 SYNC PULSE STATE
	//
	//===============================//
	//
	case RX_STATE_V1_SYNC_PULSE:
		// 
		// this should be the end of the long sync pulse
		//
		if (rf_was_on && (captured_period >= v1_sync_on[0]) && (captured_period <= v1_sync_on[1]))
		{
			v1_sync_end = captured_time;
			rx_state = RX_STATE_V1_SYNC_END;
			RX_LED_ON;			
		}
		else
		{
			UNSAFE_WEATHER_RESET();
		}
		break;
	//
	//===============================//
	//
	//         V1 SYNC END
	//
	//===============================//
	//
	case RX_STATE_V1_SYNC_END:
		//
		// this should be the end of the long off period following the long sync pulse.
		// it can have two legal ranges of values -- a short duration and a long duration
		//
		short_period = long_period = false;
		
		if (!rf_was_on)		
		{
			short_period = (captured_period >= v1_sync_end_short[0]) && (captured_period <= v1_sync_end_short[1]);
			if (!short_period)
			{
				long_period =
					((captured_period >= v1_sync_end_long[0]) && (captured_period <= v1_sync_end_long[1]));
			}
		}
		
		if (!short_period && !long_period)
		{
			UNSAFE_WEATHER_RESET();
			break;
		}
		//
		// the off period has a valid duration. perform setup for receiving the data payload
		// and transition to the receiving state.
		//
		half_clock_count = short_period ? 1 : 2;
		packet[0] = 0;
		bufptr.value = short_period ? 0 : 1;

		RX_LED_ON;
		LOCK_LED_OFF;
		rx_state = RX_STATE_RECEIVING_V1;
		protocol_version = 1;
		//
		// read ADCH to clear the ADC registers because the timer event
		// will be the end of a long RF pulse and we'll use that to measure RSSI
		//
#if WX_SHIELD_VERSION == 2
		adc_high = ADCH;
		capture_rssi = 2;
#endif

		break;
	//
	//===============================//
	//
	//         RECEIVING STATE(S)
	//
	//===============================//
	//
#if ACCEPT_SQUARE_WAVE
	case RX_STATE_RECEIVING_SQ:
#endif
	case RX_STATE_RECEIVING_V1:
	case RX_STATE_RECEIVING_V2:
	case RX_STATE_RECEIVING_V3:  

#if WX_SHIELD_VERSION == 2
		//
		// read the ADC to get RSSI value if necessary
		//
		if (capture_rssi > 1)
		{
			if (!rf_was_on)
			{
				// this is the beginning of an RF pulse, so enable RSSIC
				// by changing PORTC bit 1 to an input.
				DDRC &= ~0x02;
				capture_rssi = 1;
			}
		}
		else if (capture_rssi)
		{
			if (rf_was_on)
			{
				// this is the end of the RF pulse we used to measure RSSI.
				// disable RSSIC by changing PORTC bit 1 to an output (which has
				// already been programmed to the LOW state).
				//
				DDRC |= 0x02;
				//
				// start an ADC at the end of the rf pulse. if we try to start it at the beginning,
				// the RSSIOUT voltage will not yet reflect the pulse amplitude so we wait until the
				// end. 
				//
				adc_sra = ADCSRA & ~0x10; // get current value and turn off the interrupt status bit
				ADCSRA = adc_sra | 0x10;  // clear the conversion complete bit
				ADCSRA = adc_sra | 0x40;  // write a 1 to the start conversion bit
				ADCSRA = adc_sra;         // set the start bit back to zero (is this necessary?)
			}
			else
			{
				adc_sra = ADCSRA;
				if (adc_sra & 0x10)
				{
					adc_low = ADCL;
					adc_high = ADCH;
					rssi = (adc_high << 8) | adc_low;
				}
				else
				{
					// the conversion did not finish so set RSSI to zero to flag the problem
					//
					adc_low = 0;
					adc_high = 0;
					rssi = 0;
				}
				capture_rssi = 0;
			}
		}
#endif

		if (!short_period && !long_period)
		{
			//
			// transition periods outside the valid ranges for long or short periods occur in two
			// situations: (a) a new message has begun before timer2 can produce a timeout to end
			// the first message, and (b) data corruption through interference, RF noise, etc.
			// In both cases, if there are enough bits to form a message, signal a message complete
			// status and let the downstream software try to decode it. Otherwise, perform a reset
			// and wait for another message.
			//
			if ((bufptr.value > 40) || (protocol_version == 1 && bufptr.value > 24))
			{
				rx_state = RX_STATE_PACKET_RECEIVED;				
				RX_LED_OFF;
			}
			else
			{
				RX_LED_OFF;
				LOCK_LED_ON;
				lock_led_on_time = captured_time;
				UNSAFE_WEATHER_RESET();
			}
			break;
		}
		//
		// update our notion of half-time or time counted in half-clock periods.
		//
		half_clock_count += short_period ? 1 : 2;
		
		if (half_clock_count & 0x01)
		{
			//
			// we are at the boundary between two clock periods. no bit information is
			// available here. however, we can check to make sure there was a transition
			// at the middle of the last clock period (this is an error check only).
			//
			if (long_period)
			{
				//
				// this means there was no transition in the middle of the last clock period.
				// that is illegal for this modulation format.
				//
				RX_LED_OFF;
				LOCK_LED_ON;
				lock_led_on_time = captured_time;
				UNSAFE_WEATHER_RESET();
			}
		}
		else
		{
			//
			// we are at the middle of a clock period and a new bit value is available.
			// handle the duplicated bits in version 2.1 messages first
			//
			if (protocol_version == 2)
			{
				//
				// unless we are at the second bit in a bit pair, this bit is not kept.
				//
				dump_bit = (half_clock_count & 0x03) != 0;

				if (!dump_bit && short_period)
				{
					//
					// this means that the last two bits in this bit pair were identical.
					// version 2.1 protocol always sends the two bits in a pair inverted
					// from one another so this is an error.
					//
					RX_LED_OFF;
					LOCK_LED_ON;
					lock_led_on_time = captured_time;
					UNSAFE_WEATHER_RESET();
				}
			}
			else
			{
				dump_bit = false;
			}
			//
			// detect a bit for all protocol versions now
			//
			if (!dump_bit)
			{
				if(rf_was_on)
					packet[bufptr.portion.nibble] |= 1 << bufptr.portion.bit;          
				else 
					packet[bufptr.portion.nibble] &= ~(1 << bufptr.portion.bit);				

				bufptr.value++;					
			}			

#if ACCEPT_SQUARE_WAVE
			if (bufptr.value >= SQ_MSG_LEN)
			{
				rx_state = RX_STATE_PACKET_RECEIVED;
				RX_LED_OFF;
			}
#endif

		}
		break;
	//
	//===============================//
	//
	//     PACKET RECEIVED STATE
	//
	//===============================//
	//
	case RX_STATE_PACKET_RECEIVED:		
		//
		// this most often will happen when a new message begins before the background loop has
		// had a chance to read the current message. in this situation, we just let the new message
		// bits go into the bit bucket...
		//
		want_agc_frozen = false; // for safety -- just in case nobody has cleared it yet
		break;
	//
	//===============================//
	//
	//         DEFAULT STATE
	//
	//===============================//
	//
	default:
		UNSAFE_WEATHER_RESET();
		break;
	}
	//
	//===============================//
	//
	//      END OF CASE STATEMENT
	//
	//===============================//
	//

	previous_captured_time = captured_time;

	// guard against buffer overflows this could happen if two messages overlap
	// just right -- probably very rare.
	if (bufptr.portion.nibble >= (MAX_MSG_LEN - 2))
	{
		LOCK_LED_ON;
		lock_led_on_time = captured_time;
		UNSAFE_WEATHER_RESET();
	}

	if ((rx_state == RX_STATE_RECEIVING_V2) || (rx_state == RX_STATE_RECEIVING_V3)) 
	{
		// when waiting for another transition, set timer 2 for a timeout
		// in case we have reached the end of the message (2.048msec timeout)
		TCCR2B = T2_PRESCALE_X128;
		// TCNT2 = 64U;
		TIFR2 = PULSE_TIMEOUT_FLAG; // clear overflow bit before enabling interrupts
		TIMSK2 = PULSE_TIMEOUT_ENABLE;
	}
	else 
	{
		//
		// be sure to set timeouts when the AGC is frozen
		//
		if (want_agc_frozen)
		{
#if DETECT_PSM
			if (rx_state == RX_STATE_AR_PREAMBLE || rx_state == RX_STATE_RECEIVING_AR)
			{
				// the entire PSM message requires a long timeout (16 msec).
				//
				TCCR2B = T2_PRESCALE_X1024;
			}
			else
#endif
			{
				// if want_agc_frozen and we are not in any of the above states, we must be 
				// receiving a V1 message (that's sort of indirect logic, so be careful when
				// adding more states which use frozen agc). todo: modify this to test for a V1
				// state instead of just assuming it.
				//
				// for the sync portion of the message a long timeout is necessary (16msec).
				// for the data portion, a shorter one will work (4msec).
				// if the rx state is receiving v1 then it's the data portion, else the sync portion
				//
				TCCR2B = (rx_state == RX_STATE_RECEIVING_V1) ? T2_PRESCALE_X256 : T2_PRESCALE_X1024;
			}
			TCNT2 = 0U;
			TIFR2 = PULSE_TIMEOUT_FLAG; // clear overflow bit before enabling interrupts
			TIMSK2 = PULSE_TIMEOUT_ENABLE;
		}		
	}
}

void wxrx_enable(byte enable)
{
	noInterrupts();
	if (enable)
	{
		//
		// should we attempt to clear any pending interrupts first...?
		// yes, this is mandatory or the capability to write SPI data 
		// to the receiver while it is in the middle of a message will not work.
		//
		TIFR1 = 0x20;
		//
		// enable timer 1 interrupts for input capture events only.
		//
		TIMSK1 = (_BV(ICIE1)) | (_BV(TOIE1));
	}
	else
	{
		TIMSK1 = 0;
	}
	interrupts();
}

void wxrx_init()
{
	noInterrupts();

	want_agc_frozen = false;

	int i;
	// we never write to the high nibble of packet[] elements so clear them all now.
	for (i=0; i<MAX_MSG_LEN; packet[i++] = 0);

	WEATHER_RESET();
	// 
	// configure ports:
	//
	// ports are initialized with all pins as inputs. 
	// when changed to outputs, all bits are defaulted to zero
	// all pullup resistors are disabled by default
	//
	// setup the bits to control the LEDs.
	// turn off the LEDs.
	//
#if ENABLE_LEDS

	pinMode(RECEIVING_LED_PIN, OUTPUT);
	pinMode(LOCKED_LED_PIN, OUTPUT);
	//
	// Flash the leds a bit to let them know we're alive
	//
	for (i=0; i<10; i++)
	{
		RX_LED_ON;
		delay(50);
		RX_LED_OFF;
		LOCK_LED_ON;
		delay(50);
		LOCK_LED_OFF;
		delay(50);
	}
#endif

	//
	// Setup RSSI pins
	//
#if WX_SHIELD_VERSION == 2
	pinMode(RSSIC_PIN, INPUT);
	digitalWrite(RSSIC_PIN, LOW);
	pinMode(RSSIC_PIN, OUTPUT);
	digitalWrite(RSSIC_PIN, LOW);
#endif
	//
	// setup timer 1 to be triggered by transitions on the DATA  line
	// from the 433.92MHz receiver module. Arduino uses timer 1 for PWM
	// so this code will not work at the same time as some PWM applications
	//
	TCCR1A = 0; // Select normal (simple count-up) mode
	//
	// ICNC1 enables the noise canceller for timer event captures
	// CS10/11/12 bits select the timer clock to be the system clock/64
	// for the Duemillinova, this is 16MHz/64 or 250kHz.
	//
	TCCR1B = ( _BV(ICNC1) | _BV(CS11) | _BV(CS10) );
	SET_INPUT_CAPTURE_RISING_EDGE();
	//
	// disable timer 1 interrupts to start with.
	//
	TIMSK1 = 0; // (_BV(ICIE1)) | (_BV(TOIE1));
	// 
	// setup timer 2 to provide a timeout when waiting for receiver DATA transitions
	// timer2 is also used for PWM by Arduino, so this code may not be compatible with
	// some PWM applications.
	//
	TCCR2A = 0; // select normal mode for the timer
	TCCR2B = T2_PRESCALE_X128;  // select clk/128. 8-bit timer will overflow every 2 msec
	TCNT2 = 0;  // clear the timer count
	TIMSK2 = PULSE_TIMEOUT_DISABLE; // interrupts are disabled to start with...

#if WX_SHIELD_VERSION == 2
	//
	// setup the ADC to read channel 0, triggered by timer capture events (RSSI input)
	//
	DIDR0 = 0x01; // disable digital i/o circuitry on ADC0 input pin
	ADMUX=0x00;   // Select AREF (external input) as Vref, right justified data and channel 0 for input
	ADCSRB=0x07;  // Select timer 1 capture event as trigger source (we don't use that anymore, however)
	//
	// enable ADC but do not start any conversions yet. prescaler is 128 (125kHz conversion rate)
	//
	ADCSRA=0x97;  //0xB7; 

#if ENABLE_HDWR_DEBUG_FLAGS
	pinMode(DEBUG_FLAG_1_PIN, OUTPUT);
	pinMode(DEBUG_FLAG_2_PIN, OUTPUT);
	digitalWrite(DEBUG_FLAG_1_PIN, LOW);
	digitalWrite(DEBUG_FLAG_2_PIN, LOW);
#endif

#endif

	interrupts();
}

boolean wxrx_data_available()
{
	return rx_state == RX_STATE_PACKET_RECEIVED;
}

byte get_wxrx_data(byte *buffer, byte length, byte *protocol)
{
	if (rx_state != RX_STATE_PACKET_RECEIVED) return 0;
	*protocol = protocol_version;
	if (bufptr.portion.bit != 0)
	{
		// bump up to an even multiple of 4 bits
		bufptr.value += 4 - bufptr.portion.bit;
	}
	byte cnt = bufptr.portion.nibble;
	if (length < cnt) return 0;
	memcpy(buffer, packet, cnt);
	return cnt;
}

void start_receiving()
{
	if (rx_state != RX_STATE_PACKET_RECEIVED) return;
	WEATHER_RESET();
}

#if WX_SHIELD_VERSION == 2

unsigned int get_current_rssi() { return rssi; }

unsigned int sample_rssi()
{
	adc_sra = ADCSRA & ~0x10;
	ADCSRA = adc_sra | 0x10; // clear conversion complete bit
	ADCSRA = adc_sra | 0x40; // start a conversion
	ADCSRA = adc_sra;		 // start bit back to zero (is this necessary?)

	// wait for conversion complete or a timeout

	unsigned int start_time = ICR1;
	unsigned int stop_time = ICR1 + 60;
	unsigned int ss = 0;

	do
	{
		adc_sra = ADCSRA;
		if (adc_sra & 0x10) 
		{
			byte adc_low = ADCL;
			byte adc_high = ADCH;
			ss = (adc_high << 8) | adc_low;
			break;
		}
	} while (MILLIS_CMP(stop_time, ICR1) > 0);

	return ss;
}

boolean need_agc_frozen()
{
	return want_agc_frozen;
}

void clear_agc_freeze_request()
{
	want_agc_frozen = false;
}

#endif


#if ENABLE_HDWR_DEBUG_FLAGS
void flip_debug_flag_1()
{
	byte state = digitalRead(DEBUG_FLAG_1_PIN);
	digitalWrite(DEBUG_FLAG_1_PIN, state ? BIT_ZERO : BIT_ONE);
}
#endif
