//============================================================================
// Weather Shield for Weather Station Data Logger  
// 
//
//This application is free software; you can redistribute it and/or
//modify it under the terms of the GNU Lesser General Public
//License as published by the Free Software Foundation; either
//version 3 of the License, or (at your option) any later version.
//
//This application is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR When PARTICULAR PURPOSE. See the GNU
//Lesser General Public License for more details.
//
//You should have received a copy of the GNU Lesser General Public
//License along with this library; if not, see <http://www.gnu.org/licenses/>
//
//=============================================================================

// this is the header file for the low-level receiver firmware which 
// operates with the Weather Shield for WSDL (versions 1 and 2) and uses the 
// either the SparkFun RX3400 or Freescale MC33596 receiver.

//
// this will allow the receiver to accept a simple 1024Hz square wave modulated
// OOK signal as valid. Useful for testing and evaluating receiver performance.
//
#define ACCEPT_SQUARE_WAVE 0

#if ACCEPT_SQUARE_WAVE
#define SQ_HDR_CNT 64   // how many cycles count as a header?
#define SQ_MSG_LEN 48   // how many cycles required to form a "message"?
#endif

// Maximum number of nibbles in a message. Determines buffer size.
// maximum message length in bits is four times this value
// maximum known message is 23 nibbles, when quadrupled as would be the case in a
// v2.1 message, this could be 4 * 23 + 8 (2nd msg preamble of 32 bits) = 100 nibbles.
#define MAX_MSG_LEN                 120

//
// MILLIS_CMP(a,b) returns the sign of (a-b) -- or zero if a and b are identical
// in essence, this does a signed comparison of unsigned long numbers and makes the assumption
// that when two numbers differ by more than mm_diff, that an overflow or underflow must have 
// occurred. the over/underflow is "fixed" and the proper answer is returned
//
#define MILLIS_CMP(a,b) ( (a==b) ? 0 : ( (a>b) ? (((a-b)>mm_diff) ? -1 : 1) : (((b-a)>mm_diff) ? 1 : -1) ) )

#if WX_SHIELD_VERSION == 2
//
// driving this pin low places the MC33596 in configuration mode and enables the 
// level translators (buffers) for the SPI bus. This is the normal SPI SS pin in the ATmega328
// and it is best configured as an output in master SPI mode.
//
#define SPI_RX_SELECT_PIN   10
//
// the RSSIC pin is used to request a signal strength measurement and is connected to
// the pin by the same name on the MC33596 receiver.
//
#define RSSIC_PIN           15 /* Analog 1 -- port C bit 1 */

#endif

#if WX_SHIELD_VERSION == 2
//
// There's a red LED ("LOCKED") and a green LED ("RECEIVING") on some shield boards.
// The green LED is used to indicate that a message is in the process of being received,
// while the red LED is turned on to indicate that a corrupt message has been received.
//
#define RECEIVING_LED_PIN   5
#define LOCKED_LED_PIN      4
#define DEBUG_FLAG_1_PIN	2
#define DEBUG_FLAG_2_PIN	3
#endif

#if WX_SHIELD_VERSION == 1

#define RECEIVING_LED_PIN   6
#define LOCKED_LED_PIN      7

#endif

#if WX_SHIELD_VERSION == 2
//
// bit definitions for read or write operations on the SPI bus
//
#define MC33596_READ   0x00
#define MC33596_WRITE  0x01
//
// for now only bank-A registers are defined since bank-B is not used
// these definitions only write one register at a time. later, multiple
// write commands could be tried. for example, a value of 0x48 would
// write to both frequency registers at once.
//
#define MC33596_CONFIG1A    0x00
#define MC33596_CONFIG2A    0x02
#define MC33596_CONFIG3A    0x04
#define MC33596_CMDA        0x06
#define MC33596_FREQ1A      0x08
#define MC33596_FREQ2A      0x0A
#define MC33596_RXONOFFA    0x12
#define MC33596_HEADERA     0x14
#define MC33596_IDA         0x16
#define MC33596_RSSIA       0x18
//
// some default values for the important registers
//
#define MC33596_CONFIG1_RESET     0x54
#define MC33596_CONFIG1_DFLT      0x50  /* for the 434MHz band */
#define MC33596_CONFIG1_800MHZ    0xF0
#define MC33596_CONFIG2_DFLT      0xC4    /* 6kHz IF bandwidth */
#define MC33596_CONFIG2_FREEZE_DS 0x44 /* freezes data slicer threshold for slow data rate signals */
//
// this register contains bits that set the RF attenuation
//
// values for atten: 0=0dB, 4=8dB, 8=16dB, C=30dB
//
#define MC33596_CONFIG3_DFLT  0x00    /* includes 0dB of RF attenuation */
#define MC33596_RF_ATTEN_0DB   0x00
#define MC33596_RF_ATTEN_8DB   0x04
#define MC33596_RF_ATTEN_16DB  0x08
#define MC33596_RF_ATTEN_30DB  0x0C

#define MC33596_CMD_DFLT       0x11    /* select Bank A, Enable RSSI function */
#define MC33596_CMD_FREEZE_AGC 0x13    /* freezing AGC for slow data rate transmissions */
//
// frequency 433.92MHz + 1.5MHz IF. See the data MC33596 data sheet for forumlas
// to compute this value
//
#define MC33596_FREQ1_DFLT    0x0A
#define MC33596_FREQ2_DFLT    0x48

#define MC33596_CONFIG3_BIT_LOCK 0x20  /* PLL lock status */

#endif  // shield version 2

void wxrx_init();
void wxrx_enable(byte enable);
void start_receiving();
void led_update_check();

boolean wxrx_data_available();

//
// returns count of bytes copied into buffer
//
byte get_wxrx_data(byte *buffer, byte length, byte *protocol_version);

unsigned int sample_rssi();

#if WX_SHIELD_VERSION == 2

unsigned int get_current_rssi();

//void flip_debug_flag_1();

boolean need_agc_frozen();
void clear_agc_freeze_request();

#endif

