//============================================================================
//Weather Station Data Logger : Weather Shield for Arduino
//Copyright  2010, Weber Anderson
// 
//This application is free software; you can redistribute it and/or
//modify it under the terms of the GNU Lesser General Public
//License as published by the Free Software Foundation; either
//version 3 of the License, or (at your option) any later version.
//
//This application is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR When PARTICULAR PURPOSE. See the GNU
//Lesser General Public License for more details.
//
//You should have received a copy of the GNU Lesser General Public
//License along with this library; if not, see <http://www.gnu.org/licenses/>
//
//=============================================================================

#include "sht1x.h"
#include "Sensirion.h"

const unsigned long mm_diff = 0x7fffffffUL;

Sht1xSensors::Sht1xSensors()
{
  sht1x_clock_pin = DEFAULT_SHT1X_CLK_PIN;
  sht1x_data_pin = DEFAULT_SHT1X_DATA_PIN;
}

void Sht1xSensors::init(byte clk_pin, byte data_pin)
{
  sht1x_clock_pin = clk_pin;
  sht1x_data_pin = data_pin;
  init();
}

void Sht1xSensors::init()
{
  //
  // setup pins for the SHT1x temp/RH sensor.
  //
  noInterrupts();
  pinMode( sht1x_clock_pin, OUTPUT );
  pinMode( sht1x_data_pin, INPUT );     // redundant as pins default to inputs. here for safety
  digitalWrite( sht1x_clock_pin, LOW );
  digitalWrite( sht1x_data_pin, HIGH ); // enable pull-up resistor
  interrupts();
  msmt_rqst = false;
  recover();
}

void Sht1xSensors::recover()
{
  // don't clear msmt_rqst in this function -- if there's a request,
  // it will be re-tried after the state machine starts up again.
  interface_reset();
  state = SHT1X_STATE_INIT;
  data_ready = false;
  crc_failed = false;
  ack_failed = false;
  bad_crc_count = 0;
}

boolean Sht1xSensors::dataReady() { return data_ready; }

boolean Sht1xSensors::dataValid() { return !crc_failed; }

void Sht1xSensors::start_measurement()
{
  crc_failed = false;
  data_ready = false;
  msmt_rqst = true;
}

//
// none of the SHT1x I/O is interrupt driven. This interface is a total
// bit banger with no delays required so interrupts will not be of much use.
// The SHT1x's interface logic is all static, so it won't hurt anything if
// we get interrupted during an operation -- as long as the interrupt does not
// mess around with our I/O lines!
//
void Sht1xSensors::interface_reset()
{
  SHT1X_DATA_HIGH;
  for (int k=0; k<10; k++)
  {
    SHT1X_CLOCK_HIGH;
    SHT1X_CLOCK_LOW;
  }    
}

boolean Sht1xSensors::send_cmd(byte cmd)
{
  // transmission start sequence
  SHT1X_CLOCK_HIGH;
  SHT1X_DATA_LOW;
  SHT1X_CLOCK_LOW;
  SHT1X_CLOCK_HIGH;
  SHT1X_DATA_HIGH;
  SHT1X_CLOCK_LOW;  

  // clock out the bits...
  boolean datastate = true; // current state of data pin
  for (int k=0; k<8; k++)
  {
    // set the DATA pin output to match the data MSB
    // to avoid unnecessary noise on the DATA pin, only change the 
    // pin state if it is not equal to the desired state.
    //
    boolean da_bit = ((cmd & 0x80) == 0x80);
    boolean state_ok = datastate == da_bit;
    if (!state_ok)
    {
      if (da_bit)
      {
        SHT1X_DATA_HIGH;
      }
      else 
      {
        SHT1X_DATA_LOW;
      }
      datastate = da_bit;
    }
    // toggle the clock
    SHT1X_CLOCK_HIGH;
    SHT1X_CLOCK_LOW;

    cmd = cmd << 1; // shift the next output bit into the MSB position
  }
  //
  // after sending 8 bits, the SHT1x sensor will drive the data line low as an
  // acknowledgement of the command.
  //
  if (!datastate) 
  {
    SHT1X_DATA_HIGH; // prepare to read
    datastate = true;
  }
  boolean ack = digitalRead(sht1x_data_pin) == LOW;
  // toggle clock once more to clear the ACK
  SHT1X_CLOCK_HIGH;
  SHT1X_CLOCK_LOW;

  if (!ack)
  {
    return false;
  }

  // verify that data has gone high again -- should we do this?
  if (digitalRead(sht1x_data_pin) == LOW) return false;

  return true;
}

void Sht1xSensors::read_data(unsigned int *value, byte *crc)
{
  // caller must ensure data is ready to be read before calling this function
  // first data bit is ready to be read upon entry to this routine.
  byte x[3];
  byte k;
  byte m;
  byte d;

  for (m=0; m<3; m++)
  {
    x[m] = 0;
    for (k=0; k<8; k++)
    {
      x[m] = (x[m] << 1);
      if (digitalRead(sht1x_data_pin) == HIGH) x[m] |= 0x01U;
      SHT1X_CLOCK_HIGH;
      SHT1X_CLOCK_LOW;        
    }
    // got the byte. send an ACK with data low
    // except for the last byte, which has an ACK with data high
    if (m < 2) 
    {
      SHT1X_DATA_LOW;
    }
    SHT1X_CLOCK_HIGH;
    SHT1X_CLOCK_LOW;
    if (m < 2) 
    {
      SHT1X_DATA_HIGH;
    }
  }

  *value = ((unsigned int)x[0] << 8) | (unsigned int)x[1];
  *crc = x[2];
}
//
// this is from the Sensirion application note on CRC-8 calculation.
// it has been modified slightly to generate the CRC in the proper
// bit order, matching the ordering of bits in the CRC reported by the sensor.
//
// the crc computation includes the command byte that was sent
// in addition to the 16-bit data returned from the command. this provides
// an extra level of confirmation that the sensor is responding to 
// the proper command.
//
boolean Sht1xSensors::verify_crc8(byte cmd, unsigned int value, byte crc)
{
  byte reg = 0;
  byte xmit[3];
  xmit[0] = cmd;
  xmit[1] = value >> 8;
  xmit[2] = value & 0xff;
  boolean b0, b;

  int k,m;
  for (m=0; m<3; m++)
  {
    for (k=0; k<8; k++)
    {
      b = (xmit[m] & 0x80) == 0x80;
      b0 = (reg & 0x01) == 0x01;
      reg = reg >> 1;
      if (b != b0)      
      {
        reg |= 0x80;
        reg ^= 0x0C;
      }
      xmit[m] = xmit[m] << 1;
    }
  }
  return crc == reg;
}

boolean Sht1xSensors::getRawData(unsigned int *temp, unsigned int *rh)
{
  *temp = rawTemp;
  *rh = rawRh;
}

// corrected data is nice, but including a single floating point
// operation in any file causes inclusion  of the floating point library,
// which is over 1k bytes in size, so omitting this really saves
// some code space.

#if ENABLE_CORRECTED_DATA

boolean Sht1xSensors::getCorrectedData(float *temp, float *rh)
{
  if (!data_ready || crc_failed) return false;
  *temp = tempDegC;
  *rh = rhPct;
}

//
// these functions are from the SHT1x data sheet
// there is a polynomial to convert temperature readings into 
// temperature.
//
float Sht1xSensors::rdg2temp(unsigned int rdg)
{
  float g = (float)rdg - 7000.0F;
  float t = -40.1 + 0.01 * (float)rdg - 2.0e-8 * g * g;
  return t;
}
//
// For RH, there is a polynomial to convert readings into % RH
// There is also a smaller correction based on the temperature
//
float Sht1xSensors::rdg2rh(unsigned int rdg, float temp)
{
  float so = (float)rdg;
  float rh_linear =  -2.0468F + so * (0.0367 -1.5955e-6 * so);
  float corr = (temp - 25.0F) * (0.01 + 0.00008 * so);
  return rh_linear + corr;
}

#endif

//
// This is currently designed to alternate readings of temperature and RH, 
// one reading every 10 seconds. The delays are hard-coded below and readings 
// could easily be taken more often and averaged if desired. Sensirion data 
// sheet notes that in high res mode (14-bit), no more than 1 msmt every
// 3.2 seconds should be made to keep self-heating below 0.1 degC.
//
void Sht1xSensors::step()
{
  unsigned long now;
  unsigned int data;
  byte crc;
  char hexMsg[5];
  float x;

  switch (state)
  {
  case SHT1X_STATE_INIT:
    delay_timeout = millis() + 1500UL;
    state = SHT1X_STATE_IDLE;
    break;

  case SHT1X_STATE_IDLE:    
    if (!msmt_rqst) break;
    if (bad_crc_count > CRC_ERROR_RESET_THRESHOLD) 
    {
      recover();
      break;
    }

    now = millis();
    if (MILLIS_CMP(now, delay_timeout) == 1)
    {
      ack_failed = !send_cmd( SHT1X_MEAS_TEMP_CMD );
      state = ack_failed ? SHT1X_STATE_FAIL : SHT1X_STATE_READ_TEMP;
    }
    break;

  case SHT1X_STATE_READ_TEMP:
    if (digitalRead(sht1x_data_pin) == LOW)
    {
      read_data(&data, &crc);
      if (verify_crc8( SHT1X_MEAS_TEMP_CMD, data, crc ))
      {
        rawTemp = data;
#if ENABLE_CORRECTED_DATA
        tempDegC = rdg2temp(data);
#endif
      }
      else
      {
        crc_failed = true;
        bad_crc_count++;
        msmt_rqst = false;
        data_ready = true;
        state = SHT1X_STATE_IDLE;
      }
      delay_timeout = millis() + 10000UL;
      state = SHT1X_STATE_WAIT;
    }
    break;

  case SHT1X_STATE_WAIT:
    now = millis();
    if (MILLIS_CMP(millis(), delay_timeout) == 1)
    {
      ack_failed = !send_cmd(SHT1X_MEAS_RH_CMD);
      state = ack_failed ? SHT1X_STATE_FAIL : SHT1X_STATE_READ_RH;
    }
    break;

  case SHT1X_STATE_READ_RH:
    if (digitalRead(sht1x_data_pin) == LOW)
    {
      read_data(&data, &crc);
      data_ready = true;
      msmt_rqst = false;
      
	  if (verify_crc8( SHT1X_MEAS_RH_CMD, data, crc ))
      {
        rawRh = data;
#if ENABLE_CORRECTED_DATA
        rhPct = rdg2rh(data, tempDegC);
#endif
      }
      else
      {
        crc_failed = true;
        bad_crc_count++;
      }
      delay_timeout = millis() + 10000UL;
      state = SHT1X_STATE_IDLE;
    }
    break;

  case SHT1X_STATE_FAIL:
  default:
    recover();
    break;
  }  
}


Sht1xSensors Sensirion = Sht1xSensors();

