//============================================================================
//Weather Station Data Logger : Weather Shield for Arduino
//Copyright  2010, Weber Anderson
// 
//This application is free software; you can redistribute it and/or
//modify it under the terms of the GNU Lesser General Public
//License as published by the Free Software Foundation; either
//version 3 of the License, or (at your option) any later version.
//
//This application is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR When PARTICULAR PURPOSE. See the GNU
//Lesser General Public License for more details.
//
//You should have received a copy of the GNU Lesser General Public
//License along with this library; if not, see <http://www.gnu.org/licenses/>
//
//=============================================================================

#include "sht1x.h"
#include "Sensirion.h"
#include <avr/pgmspace.h>
//
// un-comment the line below to use a bit-wise CRC computation instead of
// the faster table-driven implementation.
//
// #define USE_BITWISE_CRC_METHOD

const unsigned long mm_diff = 0x7fffffffUL;

#ifndef USE_BITWISE_CRC_METHOD

//
// this CRC table is placed into program (flash) memory because there is a lot more of that available
// than RAM and we don't ever modify the table anyway.
// for Arduino versions 1.5 and later this declaration is different...why???
//
//static prog_uchar crc8table[256] PROGMEM = {
static const unsigned char crc8table[256] PROGMEM = {
	0x00, 0x31, 0x62, 0x53, 0xc4, 0xf5, 0xa6, 0x97, 0xb9, 0x88, 0xdb, 0xea, 0x7d, 0x4c, 0x1f, 0x2e,
	0x43, 0x72, 0x21, 0x10, 0x87, 0xb6, 0xe5, 0xd4, 0xfa, 0xcb, 0x98, 0xa9, 0x3e, 0x0f, 0x5c, 0x6d,
	0x86, 0xb7, 0xe4, 0xd5, 0x42, 0x73, 0x20, 0x11, 0x3f, 0x0e, 0x5d, 0x6c, 0xfb, 0xca, 0x99, 0xa8,
	0xc5, 0xf4, 0xa7, 0x96, 0x01, 0x30, 0x63, 0x52, 0x7c, 0x4d, 0x1e, 0x2f, 0xb8, 0x89, 0xda, 0xeb,
	0x3d, 0x0c, 0x5f, 0x6e, 0xf9, 0xc8, 0x9b, 0xaa, 0x84, 0xb5, 0xe6, 0xd7, 0x40, 0x71, 0x22, 0x13,
	0x7e, 0x4f, 0x1c, 0x2d, 0xba, 0x8b, 0xd8, 0xe9, 0xc7, 0xf6, 0xa5, 0x94, 0x03, 0x32, 0x61, 0x50,
	0xbb, 0x8a, 0xd9, 0xe8, 0x7f, 0x4e, 0x1d, 0x2c, 0x02, 0x33, 0x60, 0x51, 0xc6, 0xf7, 0xa4, 0x95,
	0xf8, 0xc9, 0x9a, 0xab, 0x3c, 0x0d, 0x5e, 0x6f, 0x41, 0x70, 0x23, 0x12, 0x85, 0xb4, 0xe7, 0xd6,
	0x7a, 0x4b, 0x18, 0x29, 0xbe, 0x8f, 0xdc, 0xed, 0xc3, 0xf2, 0xa1, 0x90, 0x07, 0x36, 0x65, 0x54,
	0x39, 0x08, 0x5b, 0x6a, 0xfd, 0xcc, 0x9f, 0xae, 0x80, 0xb1, 0xe2, 0xd3, 0x44, 0x75, 0x26, 0x17,
	0xfc, 0xcd, 0x9e, 0xaf, 0x38, 0x09, 0x5a, 0x6b, 0x45, 0x74, 0x27, 0x16, 0x81, 0xb0, 0xe3, 0xd2,
	0xbf, 0x8e, 0xdd, 0xec, 0x7b, 0x4a, 0x19, 0x28, 0x06, 0x37, 0x64, 0x55, 0xc2, 0xf3, 0xa0, 0x91,
	0x47, 0x76, 0x25, 0x14, 0x83, 0xb2, 0xe1, 0xd0, 0xfe, 0xcf, 0x9c, 0xad, 0x3a, 0x0b, 0x58, 0x69,
	0x04, 0x35, 0x66, 0x57, 0xc0, 0xf1, 0xa2, 0x93, 0xbd, 0x8c, 0xdf, 0xee, 0x79, 0x48, 0x1b, 0x2a,
	0xc1, 0xf0, 0xa3, 0x92, 0x05, 0x34, 0x67, 0x56, 0x78, 0x49, 0x1a, 0x2b, 0xbc, 0x8d, 0xde, 0xef,
	0x82, 0xb3, 0xe0, 0xd1, 0x46, 0x77, 0x24, 0x15, 0x3b, 0x0a, 0x59, 0x68, 0xff, 0xce, 0x9d, 0xac
};

#endif


Sht1xSensors::Sht1xSensors()
{
	sht1x_clock_pin = DEFAULT_SHT1X_CLK_PIN;
	sht1x_data_pin = DEFAULT_SHT1X_DATA_PIN;
}

void Sht1xSensors::init(byte clk_pin, byte data_pin)
{
	sht1x_clock_pin = clk_pin;
	sht1x_data_pin = data_pin;
	init();
}

void Sht1xSensors::init()
{
	//
	// setup pins for the SHT1x temp/RH sensor.
	//
	noInterrupts();
	pinMode( sht1x_clock_pin, OUTPUT );
	pinMode( sht1x_data_pin, INPUT );     // redundant as pins default to inputs. here for safety
	digitalWrite( sht1x_clock_pin, LOW );
	digitalWrite( sht1x_data_pin, HIGH ); // enable pull-up resistor
	interrupts();
	msmt_rqst = false;
	recover();
}

void Sht1xSensors::recover()
{
	// don't clear msmt_rqst in this function -- if there's a request,
	// it will be re-tried after the state machine starts up again.
	interface_reset();
	state = SHT1X_STATE_INIT;
	data_ready = false;
	crc_failed = false;
	ack_failed = false;
	bad_crc_count = 0;
}

boolean Sht1xSensors::dataReady() { return data_ready; }

boolean Sht1xSensors::dataValid() { return !crc_failed; }

void Sht1xSensors::start_measurement()
{
	crc_failed = false;
	data_ready = false;
	msmt_rqst = true;
}

//
// none of the SHT1x I/O is interrupt driven. This interface is a total
// bit banger with no delays required so interrupts will not be of much use.
// The SHT1x's interface logic is all static, so it won't hurt anything if
// we get interrupted during an operation -- as long as the interrupt does not
// mess around with our I/O lines!
//
void Sht1xSensors::interface_reset()
{
	SHT1X_DATA_HIGH;
	for (int k=0; k<10; k++)
	{
		SHT1X_CLOCK_HIGH;
		SHT1X_CLOCK_LOW;
	}    
}

boolean Sht1xSensors::send_cmd(byte cmd)
{
	// transmission start sequence
	SHT1X_CLOCK_HIGH;
	SHT1X_DATA_LOW;
	SHT1X_CLOCK_LOW;
	SHT1X_CLOCK_HIGH;
	SHT1X_DATA_HIGH;
	SHT1X_CLOCK_LOW;  

	// clock out the bits...
	boolean datastate = true; // current state of data pin
	for (int k=0; k<8; k++)
	{
		// set the DATA pin output to match the data MSB
		// to avoid unnecessary noise on the DATA pin, only change the 
		// pin state if it is not equal to the desired state.
		//
		boolean da_bit = ((cmd & 0x80) == 0x80);
		boolean state_ok = datastate == da_bit;
		if (!state_ok)
		{
			if (da_bit)
			{
				SHT1X_DATA_HIGH;
			}
			else 
			{
				SHT1X_DATA_LOW;
			}
			datastate = da_bit;
		}
		// toggle the clock
		SHT1X_CLOCK_HIGH;
		SHT1X_CLOCK_LOW;

		cmd = cmd << 1; // shift the next output bit into the MSB position
	}
	//
	// after sending 8 bits, the SHT1x sensor will drive the data line low as an
	// acknowledgement of the command.
	//
	if (!datastate) 
	{
		SHT1X_DATA_HIGH; // prepare to read
		datastate = true;
	}
	boolean ack = digitalRead(sht1x_data_pin) == LOW;
	// toggle clock once more to clear the ACK
	SHT1X_CLOCK_HIGH;
	SHT1X_CLOCK_LOW;

	if (!ack)
	{
		return false;
	}

	// verify that data has gone high again -- should we do this?
	if (digitalRead(sht1x_data_pin) == LOW) return false;

	return true;
}

byte Sht1xSensors::read_byte(boolean msb_first, boolean ack)
{
	byte k;
	byte c;
	byte one = msb_first ? 0x01U : 0x80U;

		for (k=0; k<8; k++)
		{
			c = msb_first ? (c << 1) : (c >> 1);
			if (digitalRead(sht1x_data_pin) == HIGH) c |= one;
			SHT1X_CLOCK_HIGH;
			SHT1X_CLOCK_LOW;        
		}
		// got the byte. send an ACK with data low
		// except for the last byte, which has an ACK with data high
		if (ack) SHT1X_DATA_LOW;
		SHT1X_CLOCK_HIGH;
		SHT1X_CLOCK_LOW;
		if (ack) SHT1X_DATA_HIGH; 

		return c;
}

void Sht1xSensors::read_data(unsigned int *value, byte *crc)
{
	// caller must ensure data is ready to be read before calling this function
	// first data bit is ready to be read upon entry to this routine.
	byte x[2];
	byte k;
	byte m;
	byte d;
	byte c;
	//
	// first two data bytes are sent MSB first, but the CRC is sent LSB first,
	// so we handle that in two separate loops
	//
	for (m=0; m<2; m++) x[m] = read_byte(true, true);

	c = read_byte(false, false); // don't ack the CRC and it has LSB first unlike the data bytes

	*value = ((unsigned int)x[0] << 8) | (unsigned int)x[1];
	*crc = c;
}

#ifdef USE_BITWISE_CRC_METHOD

//
// this is from the Sensirion application note on CRC-8 calculation.
// it has been modified slightly to generate the CRC in the proper
// bit order, matching the ordering of bits in the CRC reported by the sensor.
//
// Note: there is a mistake in the Sensirion application note on CRC
// calculations for this sensor. Sensirion says the CRC polynomial is:
//     x^8 + x^5 + x^4
// This is incorrect. The actual polynomial they are using is:
//     x^8 + x^5 + x^4 + x^0
// You can see this by looking at Figure 1 in their document -- the shift
// out from bit 7 is being xor'ed not only into bits 4,5 but also into bit 0.
//
// the crc computation includes the command byte that was sent
// in addition to the 16-bit data returned from the command. this provides
// an extra level of confirmation that the sensor is responding to 
// the proper command.
//
boolean Sht1xSensors::verify_crc8(byte cmd, unsigned int value, byte crc)
{
	byte reg = 0;
	byte xmit[3];
	xmit[0] = cmd;
	xmit[1] = value >> 8;
	xmit[2] = value & 0xff;
	boolean b0, b;

	int k,m;
	for (m=0; m<3; m++)
	{
		for (k=0; k<8; k++)
		{
			b = (xmit[m] & 0x80) == 0x80;
			b0 = (reg & 0x01) == 0x01;
			reg = reg >> 1;
			if (b != b0)      
			{
				reg |= 0x80;
				reg ^= 0x0C;
			}
			xmit[m] = xmit[m] << 1;
		}
	}
	return crc == reg;
}

#else

//
// this table-driven CRC implementation is much faster, and uses a CRC table
// stored in program memory (flash).
//
boolean Sht1xSensors::verify_crc8(byte cmd, unsigned int value, byte crc)
{
	byte c = pgm_read_byte_near(crc8table + cmd);
	c = pgm_read_byte_near( crc8table + (c ^ (value >> 8  ))   );
	c = pgm_read_byte_near( crc8table + (c ^ (value & 0xff))   );
	return (c == crc);
}

#endif

boolean Sht1xSensors::getRawData(unsigned int *temp, unsigned int *rh)
{
	*temp = rawTemp;
	*rh = rawRh;
}

// corrected data is nice, but including a single floating point
// operation in any file causes inclusion  of the floating point library,
// which is over 1k bytes in size, so omitting this really saves
// some code space.

#if ENABLE_CORRECTED_DATA

boolean Sht1xSensors::getCorrectedData(float *temp, float *rh)
{
	if (!data_ready || crc_failed) return false;
	*temp = tempDegC;
	*rh = rhPct;
}

//
// these functions are from the SHT1x data sheet
// there is a polynomial to convert temperature readings into 
// temperature.
//
float Sht1xSensors::rdg2temp(unsigned int rdg)
{
	float g = (float)rdg - 7000.0F;
	float t = -40.1 + 0.01 * (float)rdg - 2.0e-8 * g * g;
	return t;
}
//
// For RH, there is a polynomial to convert readings into % RH
// There is also a smaller correction based on the temperature
//
float Sht1xSensors::rdg2rh(unsigned int rdg, float temp)
{
	float so = (float)rdg;
	float rh_linear =  -2.0468F + so * (0.0367 -1.5955e-6 * so);
	float corr = (temp - 25.0F) * (0.01 + 0.00008 * so);
	return rh_linear + corr;
}

#endif

//
// This is currently designed to alternate readings of temperature and RH, 
// one reading every 10 seconds. The delays are hard-coded below and readings 
// could easily be taken more often and averaged if desired. Sensirion data 
// sheet notes that in high res mode (14-bit), no more than 1 msmt every
// 3.2 seconds should be made to keep self-heating below 0.1 degC.
//
void Sht1xSensors::step()
{
	unsigned long now;
	unsigned int data;
	byte crc;
	char hexMsg[5];
	float x;

	switch (state)
	{
	case SHT1X_STATE_INIT:
		delay_timeout = millis() + 1500UL;
		state = SHT1X_STATE_IDLE;
		break;

	case SHT1X_STATE_IDLE:    
		if (!msmt_rqst) break;
		if (bad_crc_count > CRC_ERROR_RESET_THRESHOLD) 
		{
			recover();
			break;
		}

		now = millis();
		if (MILLIS_CMP(now, delay_timeout) == 1)
		{
			ack_failed = !send_cmd( SHT1X_MEAS_TEMP_CMD );
			state = ack_failed ? SHT1X_STATE_FAIL : SHT1X_STATE_READ_TEMP;
		}
		break;

	case SHT1X_STATE_READ_TEMP:
		if (digitalRead(sht1x_data_pin) == LOW)
		{
			read_data(&data, &crc);
			if (verify_crc8( SHT1X_MEAS_TEMP_CMD, data, crc ))
			{
				rawTemp = data;
#if ENABLE_CORRECTED_DATA
				tempDegC = rdg2temp(data);
#endif
			}
			else
			{
				crc_failed = true;
				bad_crc_count++;
				msmt_rqst = false;
				data_ready = true;
				state = SHT1X_STATE_IDLE;
			}
			delay_timeout = millis() + 10000UL;
			state = SHT1X_STATE_WAIT;
		}
		break;

	case SHT1X_STATE_WAIT:
		now = millis();
		if (MILLIS_CMP(millis(), delay_timeout) == 1)
		{
			ack_failed = !send_cmd(SHT1X_MEAS_RH_CMD);
			state = ack_failed ? SHT1X_STATE_FAIL : SHT1X_STATE_READ_RH;
		}
		break;

	case SHT1X_STATE_READ_RH:
		if (digitalRead(sht1x_data_pin) == LOW)
		{
			read_data(&data, &crc);
			data_ready = true;
			msmt_rqst = false;

			if (verify_crc8( SHT1X_MEAS_RH_CMD, data, crc ))
			{
				rawRh = data;
#if ENABLE_CORRECTED_DATA
				rhPct = rdg2rh(data, tempDegC);
#endif
			}
			else
			{
				crc_failed = true;
				bad_crc_count++;
			}
			delay_timeout = millis() + 10000UL;
			state = SHT1X_STATE_IDLE;
		}
		break;

	case SHT1X_STATE_FAIL:
	default:
		recover();
		break;
	}  
}


Sht1xSensors Sensirion = Sht1xSensors();

