//============================================================================
// Weather Shield for Weather Station Data Logger  
// 
//
//This application is free software; you can redistribute it and/or
//modify it under the terms of the GNU Lesser General Public
//License as published by the Free Software Foundation; either
//version 3 of the License, or (at your option) any later version.
//
//This application is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR When PARTICULAR PURPOSE. See the GNU
//Lesser General Public License for more details.
//
//You should have received a copy of the GNU Lesser General Public
//License along with this library; if not, see <http://www.gnu.org/licenses/>
//
//=============================================================================
//

// #define ENABLE_DEBUG_PRINT 1

// this header file defines the reciever hardware to be supported
#include "WxReceiverConfig.h"

//
// set this to "1" to disable checksum verification
// sometimes useful for analyzing messages from new sensors
// or new protocols.
//
#define NO_VERIFY_CHECKSUMS 0

#include "WxSpi.h"
#include "WxReceivers.h"

extern "C" {
#include "wxrx.h"
}

const unsigned long mm_diff = 0x7fffffffUL; 

#if WX_SHIELD_VERSION == 2
//
// command sequences for the MC33596 receiver. 
// individual commands are delimited with the SPI_END bit,
// and the last command in a sequence is identified by the SPI_LAST bit.
// omitting one of these flags will really screw things up so be careful!!!
//
const unsigned int mc33596_setup_cmds[] = {
	MC33596_CONFIG1A | MC33596_WRITE, MC33596_CONFIG1_RESET | SPI_END,
	MC33596_CONFIG1A | MC33596_WRITE, MC33596_CONFIG1_DFLT | SPI_END,
	MC33596_CONFIG2A | MC33596_WRITE, MC33596_CONFIG2_DFLT | SPI_END,
	MC33596_CONFIG3A | MC33596_WRITE, MC33596_CONFIG3_DFLT | MC33596_RF_ATTEN_0DB | SPI_END,
	MC33596_CMDA | MC33596_WRITE, MC33596_CMD_DFLT | SPI_END,
	MC33596_FREQ1A | MC33596_WRITE, MC33596_FREQ1_DFLT | SPI_END,
	MC33596_FREQ2A | MC33596_WRITE, MC33596_FREQ2_DFLT | SPI_LAST
};

const unsigned int mc33596_set_rf_atten_0dB_cmds[] = {
	MC33596_CONFIG3A | MC33596_WRITE, MC33596_CONFIG3_DFLT | MC33596_RF_ATTEN_0DB | SPI_LAST
};
const unsigned int mc33596_set_rf_atten_8dB_cmds[] = {
	MC33596_CONFIG3A | MC33596_WRITE, MC33596_CONFIG3_DFLT | MC33596_RF_ATTEN_8DB | SPI_LAST
};
const unsigned int mc33596_set_rf_atten_16dB_cmds[] = {
	MC33596_CONFIG3A | MC33596_WRITE, MC33596_CONFIG3_DFLT | MC33596_RF_ATTEN_16DB | SPI_LAST
};
const unsigned int mc33596_set_rf_atten_30dB_cmds[] = {
	MC33596_CONFIG3A | MC33596_WRITE, MC33596_CONFIG3_DFLT | MC33596_RF_ATTEN_30DB | SPI_LAST
};

const unsigned int mc33596_read_rssi_cmds[] = {
	MC33596_RSSIA | MC33596_READ, 0U | SPI_LAST
};

const unsigned int mc33596_read_config3_cmds[] = {
	MC33596_CONFIG3A | MC33596_READ, 0U | SPI_LAST
};

const unsigned int mc33596_freeze_agc_cmds[] = {
	MC33596_CMDA | MC33596_WRITE, MC33596_CMD_FREEZE_AGC | SPI_LAST
};

const unsigned int mc33596_thaw_agc_cmds[] = {
	MC33596_CMDA | MC33596_WRITE, MC33596_CMD_DFLT | SPI_LAST
};

#endif

#if ENABLE_DEBUG_PRINT
const char hexChars[16] = { 
	'0', '1', '2', '3', '4', '5', '6', '7',
	'8', '9', 'A', 'B', 'C', 'D', 'E', 'F'   };
#endif

	// 
	// see the method ::get_data() for a description of the purpose 
	// and contents of this array.
	//
    const byte cksumOffsets[6] = { 4, 7, 3, 2, 6, 5 };

	WxRx::WxRx()
	{
		running = false;
#if WX_SHIELD_VERSION == 2
		reading_rssi = false;
		reading_status = false;
		updating_agc = false;
		agc_frozen = false;
#endif
	}

	void WxRx::init()
	{
		wxrx_init();
#if WX_SHIELD_VERSION == 2
		WxSpi.declare_slave_pin(SPI_RX_SELECT_PIN);
		//
		// the MC33596 treats the first SPI clock edge as a setup event and then
		// samples data on the 2nd clock (falling) clock edge. This is SPI mode "1".
		//
		clock_rate = WxSpi.begin(400000UL, (byte)1);
#endif
		running = false;
		initialized = false;
	}

	// enables reception of RF messages

	void WxRx::start()
	{
#if WX_SHIELD_VERSION == 2
		WxSpi.begin_write(mc33596_setup_cmds);
#endif
		running = false;
		initialized = false;
	}

	// temporarily halts reception of messages

	void WxRx::pause()
	{
		wxrx_enable(0);
		running = false;
	}

	// ends a temporary halt of reception

	void WxRx::resume()
	{
		wxrx_enable(1);
		running = true;
	}



#if 0  // debugging functions
	int WxRx::state()
	{
		return wxrx_state();
	}

	unsigned long WxRx::interruptCounter()
	{
		return wxrx_ic();
	}
#endif

#if WX_SHIELD_VERSION == 2
	//
	// checks to see if the current AGC state matches the
	// desired state. If not, the receiver is temporarily taken
	// out of receive mode and an SPI transaction started to change
	// the receiver's AGC state to the desired value.
	//
#if 0 // DETECT_PSM
	void WxRx::check_agc_state()
	{
		boolean need_frozen = need_agc_frozen();

		if (need_frozen != agc_frozen)
		{
			wxrx_enable(0);
			running = false;
			updating_agc = true;
			agc_frozen = need_frozen;
			WxSpi.begin_write(need_frozen ? mc33596_freeze_agc_cmds : mc33596_thaw_agc_cmds);
		}
	}
#endif
	//
	// provides a way to request a signal strength measurement, independent
	// of RF signal reception.
	//
	void WxRx::request_rssi()
	{
		wxrx_enable(0);
		running = false;
		reading_rssi = true;
		WxSpi.begin_write(mc33596_read_rssi_cmds);
	}

	boolean WxRx::rssi_available()
	{
		return !reading_rssi;
	}

	byte WxRx::rssi_register_value()
	{
		return rssi_register;
	}

	void WxRx::request_status()
	{
		wxrx_enable(0);
		running = false;
		reading_status = true;
		WxSpi.begin_write(mc33596_read_config3_cmds);
	}

	boolean WxRx::status_available()
	{
		return !reading_status;
	}

	byte WxRx::status_value()
	{
		return status_register;
	}

	boolean WxRx::set_rf_attenuation(int dB)
	{
		const unsigned int* cmds = 0;
		switch(dB)
		{
		case 0:
			cmds = mc33596_set_rf_atten_0dB_cmds;
			break;
		case 8:
			cmds = mc33596_set_rf_atten_8dB_cmds;
			break;
		case 16:
			cmds = mc33596_set_rf_atten_16dB_cmds;
			break;
		case 30:
			cmds = mc33596_set_rf_atten_30dB_cmds;
			break;
		default:
			return false;
		}

		wxrx_enable(0); // kill the interrupts and the Atmel ADC
		running = false;
		WxSpi.begin_write(cmds);
		return true;
	}

#endif // if WX_SHIELD_VERSION == 2

	//
	// the receiver can operate in two distinct states: 
	// 1) receiving an RF transmission
	// 2) getting RSSI data from the most recent RF packet
	// this state machine handles the process of acquiring 
	// RSSI data from one of the registers in the MC33596.
	//
	void WxRx::step()
	{
		//
		// the red LOCK led can get turned on when a corrupt message is received,
		// but should not be left on too long. this call will shut it off if it
		// is on and enough time has elapsed.
		//
		led_update_check();

#if WX_SHIELD_VERSION == 2

		if (running)  // we're in the normal receiving state
		{
			//
			// this state machine does nothing during normal signal reception...unless
			// a change in agc state is requested.
			//
#if 0 //DETECT_PSM
			check_agc_state();
#endif
			return;
		}

		if (WxSpi.busy()) return; // we're waiting for the SPI transaction to complete

#if 0 //DETECT_PSM
		if (updating_agc)
		{
			//
			// it takes some time for the receiver to go from config mode back to rx mode.
			// the data sheet claims it is only 50usec or less, but we've seen invalid RF
			// bit states as long as 160usec after leaving config mode. the busy-wait delay
			// here is not ideal but seems to work.
			//
			delayMicroseconds(200);
			updating_agc = false;
			wxrx_enable(1);
		}
#endif

		if (reading_rssi)
		{
			reading_rssi = false;
			if (WxSpi.read_count() == 2)
			{
				byte btmp[2];
				if (WxSpi.read(btmp, 2))
				{
					rssi_register = btmp[1];
				}
				else
				{
					rssi_register = 0xFE;
				}
			}
			else
			{
				rssi_register = 0xFF;
			}
			clear_agc_freeze_request();
		}

		if (reading_status)
		{
			reading_status = false;
			if (WxSpi.read_count() == 2)
			{
				byte btmp[2];
				if (WxSpi.read(btmp, 2))
				{
					status_register = btmp[1];
				}
				else
				{
					status_register = 0xFE;
				}
			}
			else
			{
				status_register = 0xFF;
			}
		}

#endif

		running = true;   
		if (!initialized)
		{      
			initialized = true;
			wxrx_enable(1); // re-enables interrupts for the receiver, but does not do a weather reset.   
		}
	}

#if WX_SHIELD_VERSION == 2

	unsigned int WxRx::get_rssi()
	{
		return get_current_rssi();
	}

unsigned int WxRx::force_read_rssi(unsigned int navg)
	{
		wxrx_enable(0); // disable interrupts so we can get a valid adc reading
		// without danger of an ISR clearing the registers while we're in process
		running = false;
		unsigned long sum = 0;
		for (int k=0; k<navg; k++)
		{
			sum += sample_rssi();
		}
		return (int)(sum / ((unsigned long)navg));
	}
#endif

	void WxRx::resume_receiving()
	{
		start_receiving();  // resets the state machine
		wxrx_enable(1);     // enables interrupts
	}

	void WxRx::force_reset()
	{
		init();
		start();
	}

	boolean WxRx::data_available()
	{
		return wxrx_data_available();
	}

	byte WxRx::get_data(byte *packet, byte length, byte *protocol)
	{
		if (!wxrx_data_available()) return 0;

		byte duplicateIndex = 0;
		byte duplicateLength = 0;
		byte rf_version;
		byte msgLen = get_wxrx_data(packet, length, &rf_version);
		byte minLength = (rf_version == PROTOCOL_OS1) ? 8 : 11;
		if (!PROTOCOL_IS_OS(rf_version)) minLength = 6;

#if ENABLE_DEBUG_PRINT
		if (1)
		{
			Serial.print("Got something...");
			Serial.print((int)rf_version);
			Serial.print(",");
			Serial.println((int)msgLen);
		}
#endif
		//
		// if we're going to request RSSI register values, then don't start
		// receiving a new data packet just yet...
		//
		//start_receiving();

		//
		// validate the sync nibble. it is the same for version 2 and 3 protocols
        // for other messages we do not look for a valid sync nibble
		//
		bool msgOk;
#if ACCEPT_SQUARE_WAVE
		msgOk = true;
#else
		if (rf_version == PROTOCOL_OS2 || rf_version == PROTOCOL_OS3)
		{
			msgOk = packet[0] == 0x0A;
		}
		else
		{
			msgOk = true;
		}
#endif

		//
		// the minimum length message will have a 4-nibble ID, 1-nibble channel,
		// 2-nibble rolling code, 1-nibble status, at least one nibble of
		// sensor data and a 2-nibble checksum for a total of 11 nibbles.
		//
		if (msgLen < minLength) return 0;
		//
		// for protocol version 2, there may be two concatenated copies of the same message 
		// this is indicated if the pattern "FFFFA" occurs in the message. "FFFF" is the 
		// preamble of the 2nd message and "A" is the sync nibble. time can be saved by 
		// limiting this check to messages over 27 nibbles in length.
		//
		if (msgLen > 27 && rf_version == PROTOCOL_OS2)
		{
			int k;
			unsigned long sr = 0; // shift register
			boolean foundHdr = false;

			for (k=10; k<msgLen; k++)
			{
				sr = (sr << 4) | (unsigned long)packet[k];
				if ((sr & 0x000FFFFFUL) == 0x000FFFFAUL)
				{
					foundHdr = true;
					break;
				}
			}

			if (foundHdr) // there are two messages here
			{
				// k points to the sync nibble of the 2nd message. 
				// the last nibble of the first message is packet[k-5]
				// and the length of the first message is (k-4)
				// the 2nd message begins at "k" and is only if interest if the
				// checksum in the first message is bad. make a record of the
				// existence and location of the duplicate message if needed 
				// later on below.
				//
				duplicateIndex = k;
				duplicateLength = msgLen - k;
				msgLen = k - 4;
				//
				// if by chance, the sync nibble in the first message copy
				// is wrong, then immediately switch to the 2nd one, because
				// we've already verified that it's sync nibble is correct.
				//
				if (!msgOk && (duplicateIndex > 0))
				{
					// copy the 2nd message on top of the first, destroying it
					memcpy(packet, packet+duplicateIndex, duplicateLength);
					// reset the pointers and we're done
					duplicateIndex = duplicateLength = 0;
					msgLen = duplicateLength;
					msgOk = true;
				}
			} // if (foundHdr)

		}   // if (msgLen > 27)

		do 
		{
			if (msgOk)
			{
				if (rf_version == PROTOCOL_OS1)
				{
					//
					// version 1.0 sensors use a byte-oriented checksum instead of a nibble-oriented one.
					// there must be an even number of nibbles in these messages.
					//
					unsigned int cksumIndex = msgLen - ((msgLen & 0x01) ? 3 : 2);
					msgOk = ValidV1Checksum(packet, cksumIndex);
					if (!msgOk)
					{
						msgOk = ValidV1Checksum(packet, ++cksumIndex);
					}
                    //
                    // this is tricksy. we set message length to include a non-existent 2-nibble CRC
                    // just so it looks like version 2.1 and 3.0 messages. be sure later on to remove these
                    // extra 2 nibble from consideration.
                    //
					if (msgOk) msgLen = cksumIndex + 4;
				}
				else if (PROTOCOL_IS_OS(rf_version))
				{
					//
					// attempt to validate the message checksum. There are a few variations to worry about
					// here. If the message is perfect, most sensors send 2 nibbles after the checksum, so
					// the 2-nibble checksum starts 4 nibbles before the end of the message. At least one
					// version 2.1 sensor (THR238NF) sends 5 nibbles after the checksum, so in that case
					// we would look 7 nibbles before the end to find the checksum.
					//
					// the next variable is that the end of the message might be truncated due to interference
					// (from another sensor for example). if only a few bits were lost and the checksum
					// is intact, we can still use the message. in this case, we'll need to look farther
					// along in the message to find the checksum.
					//
					// the array defined at the top of this file "cksumOffsets" contains a list of
					// offsets from the end of the message where we will look for a valid checksum.
					// the first two places are 4 and 7 nibbles from the end since these are the normal
					// locations in a good message. after that, locations for possible truncated messages
					// are searched.
					//
					unsigned int cksumIndex;				
					msgOk = false;

					for (int k=0; !msgOk && (k<sizeof(cksumOffsets)); k++)
					{
						cksumIndex = msgLen - cksumOffsets[k];
						msgOk = ValidChecksum(packet, cksumIndex);
					}

					if (msgOk) msgLen = cksumIndex + 4;				
				
					// for both version 2.1 and 3.0 protocols, msgLen is includes two nibbles
					// after the checksum, whether they were actually part of the message or not.
					// this is done so that the meaning of msgLen is consistent regardless of 
					// protocol version. it might be cleaner to make msgLen only include the checksum...?
				}
			}

			if (msgOk || (duplicateIndex == 0)) break;
			//
			// copy the duplicate message on top of the first one and see
			// if that one is valid
			//
			memcpy(packet, packet+duplicateIndex, duplicateLength);
			msgLen = duplicateLength;
			duplicateIndex = duplicateLength = 0;

		} while (true);


#if ENABLE_DEBUG_PRINT
#if !ENABLE_HEX_OUTPUT
		if (!msgOk)
#endif
		{
			// for debug, print out the message nibbles
			for (int i=0; i<msgLen; i++)
			{
				Serial.print(hexChars[packet[i]]);			
			}
			Serial.println(".");
		}
#endif

		//
		// version 2.1 protocol messages include a full repeat of the 
		// message with every transmission.
		// detect repeated version 2.1 protocol messages here
		// and get rid of one of them if it matches the previous one
		//
		if (  msgOk && ( (rf_version == PROTOCOL_OS2) || (rf_version == PROTOCOL_OS1) )  )
		{
			unsigned long now = millis();
			unsigned long tlim = previous_packet_time + 1000UL;
			if ( MILLIS_CMP(now, tlim) == -1 )
			{
				// this packet was received less than one second after the
				// previous packet, so this might be a repeated packet. 
				// the only way to know for sure is to compare data.
				// if the data is equal, memcmp will return zero.
                //
                // remove two nibbles at end from comparison if we are not 
                // checking CRC on version 2.1/3 protocols and always for
                // version 1 protocols (those 2 nibbles are "faked" for version 1 msgs)
                //
                byte clip;
#if ENABLE_CRC_OUTPUT
				clip = (rf_version == PROTOCOL_OS1) ? 2 : 0;
#else
				clip = 2;
#endif
                msgOk = memcmp(packet, previous_packet, msgLen-clip) != 0;
			}
			// log the time of this packet and save the packet data 
			previous_packet_time = now;
			memcpy(previous_packet, packet, msgLen);
		}
        //
        // for protocol versions 2.1 and 3.0 we do not transfer the first nibble 
        // why? what is it...sync?
        //
        if ((rf_version == PROTOCOL_OS2) || (rf_version == PROTOCOL_OS3))
        {
            msgLen--;
            for (int i=0; i<msgLen; i++)
            {
                packet[i] = packet[i+1];
            }
        }

		if (msgOk)
		{
			*protocol = rf_version;
			return msgLen;
		}
		else
		{
			return 0;
		}

	}

	boolean WxRx::ValidV1Checksum(byte *packet, int Pos)
	{
#if NO_VERIFY_CHECKSUMS || ACCEPT_SQUARE_WAVE
		return true;
#else
		bool ok = false;
		byte check = packet[Pos] | (byte)(packet[Pos+1] << 4);

		int checksum = 0;
		for (int x = 0; x < Pos; x += 2)
		{
			checksum += packet[x] | (packet[x+1] << 4);
		}

		checksum = (checksum & 0xff) + (checksum >> 8);

		return ((byte)checksum == check);
#endif
	}

	boolean WxRx::ValidChecksum(byte *packet, int Pos)
	{
#if NO_VERIFY_CHECKSUMS || ACCEPT_SQUARE_WAVE
		return true;
#else
		bool ok = false;
		byte check = packet[Pos] | (byte)(packet[Pos+1] << 4);

		byte Checksum = 0;
		for (int x = 1; x < Pos; Checksum += packet[x++]);	

		return (Checksum == check);
#endif
	}

	WxRx WxReceivers = WxRx();
