//============================================================================
// Weather Shield for Weather Station Data Logger  
// 
//
//This application is free software; you can redistribute it and/or
//modify it under the terms of the GNU Lesser General Public
//License as published by the Free Software Foundation; either
//version 3 of the License, or (at your option) any later version.
//
//This application is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR When PARTICULAR PURPOSE. See the GNU
//Lesser General Public License for more details.
//
//You should have received a copy of the GNU Lesser General Public
//License along with this library; if not, see <http://www.gnu.org/licenses/>
//
//=============================================================================

#include "Arduino.h"
// #include <wiring.h>
#include <avr/interrupt.h>

// this header file defines the reciever hardware to be supported
#include "WxReceiverConfig.h"

//
// to generate a version of the sketch which does not illuminate the 
// LED lamps, set this to zero to activate the empty macros.
//
#define ENABLE_LEDS					1
#define ENABLE_HDWR_DEBUG_FLAGS		0

#include "wxrx.h"

//
// macros to control and test the trigger polarity for timer 1B
//

#define INPUT_CAPTURE_IS_RISING_EDGE()    ((TCCR1B & _BV(ICES1)) != 0)
#define INPUT_CAPTURE_IS_FALLING_EDGE()   ((TCCR1B & _BV(ICES1)) == 0)
#define SET_INPUT_CAPTURE_RISING_EDGE()   (TCCR1B |=  _BV(ICES1))
#define SET_INPUT_CAPTURE_FALLING_EDGE()  (TCCR1B &= ~_BV(ICES1))
//
// macros to control the two LEDs on the weather shields
//
#if ENABLE_LEDS

#if WX_SHIELD_VERSION == 2

#define RX_LED_ON            PORTD |= _BV(RECEIVING_LED_PIN)
#define RX_LED_OFF           PORTD &= (byte)(~(_BV(RECEIVING_LED_PIN)))

#define LOCK_LED_ON          PORTD |= _BV(LOCKED_LED_PIN)
#define LOCK_LED_OFF         PORTD &= (byte)(~(_BV(LOCKED_LED_PIN)))
#define LOCK_LED_STATE	     (PORTD & _BV(LOCKED_LED_PIN))

#endif

#if WX_SHIELD_VERSION == 1

#define RX_LED_ON		digitalWrite(RECEIVING_LED_PIN, HIGH)
#define RX_LED_OFF		digitalWrite(RECEIVING_LED_PIN, LOW)

#define LOCK_LED_ON		digitalWrite(LOCKED_LED_PIN, HIGH)
#define LOCK_LED_OFF	digitalWrite(LOCKED_LED_PIN, LOW)
#define LOCK_LED_STATE  digitalRead(LOCKED_LED_PIN)

#endif

#else

#define RX_LED_ON 
#define RX_LED_OFF 
#define LOCK_LED_ON 
#define LOCK_LED_OFF 
#define LOCK_LED_STATE false

#endif
//
// some macros to make some of the code below a little more obvious
//
// macros to reset the receive state machine state. The unsafe version
// can only be used in an interrupt. 
//
#define UNSAFE_WEATHER_RESET() { rx_state = RX_STATE_IDLE; RX_LED_OFF; want_agc_frozen = false;  }  
#define WEATHER_RESET() { noInterrupts(); rx_state = RX_STATE_IDLE; RX_LED_OFF; want_agc_frozen = false; interrupts(); }  
//
// values for timer 2's control registers
//
#define PULSE_TIMEOUT_ENABLE  _BV(TOIE2)
#define PULSE_TIMEOUT_DISABLE 0
#define T2_PRESCALE_X128 ( _BV(CS22) | _BV(CS20) )
#define T2_PRESCALE_X256 ( _BV(CS22) | _BV(CS21) )
#define T2_PRESCALE_X1024 ( _BV(CS22) | _BV(CS21) | _BV(CS20) )
#define PULSE_TIMEOUT_FLAG _BV(TOV2)
//
// Preamble information
//
// this defines the minimum number of RF transitions required to identify
// a preamble sequence. Perfect version 1.0, 2.1 and 3.0 preambles contain
// 24, 32 and 24 transistions respectively.
//
// The maximum number determines when a string of pulses is not considered
// a valid preamble if they just keep going on.
//
#define V3_MIN_SYNC_COUNT           10
#define V3_MAX_SYNC_COUNT           14
#define V2_MIN_SYNC_COUNT           14
#define V2_MAX_SYNC_COUNT			18
#define V1_MIN_SYNC_COUNT			10
#define V1_MAX_SYNC_COUNT			14
//
// State machine states
//
#define RX_STATE_IDLE               0  /* ready to go */
#define RX_STATE_PACKET_RECEIVED    1  /* complete message received */
#define RX_STATE_V1_PREAMBLE		2
#define RX_STATE_V1_SYNC_PULSE		3
#define RX_STATE_V1_SYNC_END		4
#define RX_STATE_SHORT_PREAMBLE		5
#define RX_STATE_V2_PREAMBLE        6
#define RX_STATE_PSM_STABILIZE      7
#define RX_STATE_PSM_SYNC           8
#define RX_STATE_RECEIVING_PSM		9
#define RX_STATE_RECEIVING_PWM      10
#define RX_STATE_MANCHESTER         11 
#if ACCEPT_SQUARE_WAVE
#define RX_STATE_RECEIVING_SQ       12
#endif

#define BIT_ZERO                    0
#define BIT_ONE                     1



// *** RF Protocol Decoding Variables ***

//
// timer value at last edge capture event. used to compute time interval
// between previous event and current event.
//
static unsigned int timer1_ovfl_count;
static unsigned int previous_captured_time;
static unsigned int v1_sync_end;
static unsigned int previous_captured_period;
static unsigned int previous_on_period;
static unsigned int might_be_vn1tx;
static unsigned int might_be_f007th;
//
// variables used to time the illumination of the red LED.
//
static unsigned int lock_led_on_time;
//
// receive buffer pointer 
// using bit fields like this is platform dependent and generally not recommended.
// however, since this code will never (never say never :-O ) run on anything but an Atmel processor,
// the risks are minimized. on the plus side, this union with bit fields makes the code
// cleaner and easier to understand.
//
static union {
    unsigned int value;  
    struct 
    {
        unsigned int bit:2;      // two LSBs refer to a bit within the nibble
        unsigned int nibble:14;  // 14 MSBs refer to a nibble 
    } portion;
} bufptr;
//
// receive buffer
// only the lower nibble of each byte is used. The upper nibbles
// are zeroed in the reset() function and should not be modified
// thereafter.
//
static byte packet[MAX_MSG_LEN];
//
// while looking for a valid preamble, these track the number of 
// short and long pulses seen. for now, preambles are much less
// than 256 pulses long, but use 16-bit counters just to be safe
//
static unsigned int pulse_count;    // pulse count during preamble
//
// each receive state may handle more than one type of signal with different
// timing limits. these pointers are set to the appropriate timing limit
// array before entering a given receive state. not all states use sync_range
// and if inter-frame sync is not an option the pointer may be set to zero.
//
unsigned int *on_thresholds;
unsigned int *off_thresholds;
unsigned int *sync_range;           // for messages that repeat

//
// when receiving manchester-coded signals, we keep track of time in units 
// of 1/2 clock period when this value is even we are at the transition which
// defines a bit. when it is odd, we are at a transition in the middle of a bit
// period which is required to achieve the proper transition for the next bit.
//
static int half_clock_count;
//
// this is essentially an enum; 
// it should be set to values defined in WxReceiverConfig.h
//
static byte protocol_version; // protocol version of the current message
//
// for the version 2 protocol, this is used as a toggle to cause every
// other bit to be "dumped"-- bits come in inverted pairs and the first
// bit in each pair is inverted, so we dump the first bit and keep the 2nd.
//
static boolean dump_bit;

static byte rx_state; // current state of the receive state machine

//
// for this to work properly, the main run loop must call the WxRx::step()
// method periodically. This will update the receiver's AGC state according
// to the value of this variable. Also note, that only the Version 2 WxShield
// offers control of AGC state. Earlier shields cannot use this capability.
//
// =========== UPDATE ====================
//
// more recent research suggests that manipulating agc during signal reception 
// is not necessary even with version 2 shields (OS version 1.0 signals not confirmed).
//
static boolean want_agc_frozen;

static byte adc_sra;

const unsigned long mm_diff = 0x7fffffffUL; 
//
// in the idle state, we are only looking for pulses in three ranges of widths
// to make an initial guess about the probable list of sensors it might be sent from.
//
const unsigned int idle_short_range[2] = { 75, 175 }; // for a 4us tick, 300 to 700us
const unsigned int idle_os2_range[2] = { 200, 300 };   // for a 4us tick, 800 to 1200us
const unsigned int idle_os1_range[2] = { 325, 500 };   // for a 4us tick, 1300 to 2000us
//
// version 2 shields when presented with a long off period may signal the resumption of the
// rf signal before it actually happens. With a 10ms off period, the data line may go high
// between 2.5ms and 8ms but then when the RF DOES come on, the data line will stay and 
// accurately transition to low at the end of the pulse. During a PSM preamble, then if we
// accept this early transition but measure time between falling edges of data it should
// be correct. idl_psm_off_range specifies this interval.
//
// the last value in this array should equal the first value in psm_sync_off_thresholds
//
const unsigned int idle_psm_off_range[2] = { 625, 2000 }; // 2.5ms and 8ms
//
// pulse timing limits for protocol versions 2.1 and 3.0
//
// these are valid for OS manchester signals 2.1 and 3.0
// thresholds for short and long timer periods. the thresholds are different
// when the RF signal is on and off. the three values in each array are, in order:
// minimum short period, threshold between short and long periods, and maximum long period.
// if the timer value is exactly equal to the short/long threshold it does not matter 
// which choice is made -- this is probably a noise reading anyway. 
// periods less then the minimum short or maximum long periods are considered noise and
// rejected.
//
const unsigned int rf_off_thresholds[3] = { 100, 212, 350 };  // for a 4usec timer tick, 400,848,1400 usec
const unsigned int rf_on_thresholds[3]  = {  50, 154, 275 };  // for a 4usec timer tick, 200,616,1100 usec
//
// timing limits for protocol version 1.0
// lucky for us, the minimum on period in the preamble (1500usec) is longer than the maximum
// on period for both version 2.1 and 3.0 protocols which allows us to detect version 1.0 messages instantly
// in the first on pulse.
//
// Update: 12/27/2012, changed several version 1 time limits to account for additional variability in
// timing between individual sensors.
//
const unsigned int v1_off_thresholds[3] = { 243, 488, 775 }; //  970,1950,3100 usec
const unsigned int v1_on_thresholds[3]  = { 351, 600, 850 }; // 1404,2400,3400 usec
const unsigned int v1_sync_off_range[2] = { 1000, 1150 };          // 4000,4600 usec
const unsigned int v1_sync_on_range[2]  = { 1350, 1496 };          // 5400,5985 usec
const unsigned int v1_sync_end_short[2] = { 1250, 1400 };    // 5000,5600 usec
const unsigned int v1_sync_end_long[2]  = { 1625, 1775 };	 // 6500,7100 usec

const unsigned int psm_on_thresholds[3]   = { 75, 160, 161 };		// 300, 600 usec for the on pulses
//
// there are two sync-off ranges here that could be used to distinguish between sensors
// for now, we only use the first and last values.
// first range is for the 0964TX unit, 2nd range for the towers
//
const unsigned int psm_sync_off_thresholds[3]   = { 2000, 2125, 2500 };			// 8.0, 8.5, 10 msec
const unsigned int psm_off_thresholds[3]        = { 425, 750, 1150 };		// 1700, 3000, 4600 usec

//
// note: the low-end limits need to be a bit lower for the version 1 WxShields than for
// version 2 and these values seem to work well for both versions.
//
const unsigned int vn1tx_sync_on_range[2]  = { 125, 180 };         // 500 to 700us
const unsigned int vn1tx_thresholds[3]     = { 40, 75, 120 };      // 160, 300, 480us
//
// for detection of VN1TX signals at end of preamble we use tighter limits on pulse width to 
// avoid confusion with OS3 signals. important to keep upper end of long long range as low as possible
//
const unsigned int vn1tx_short_range[2]     = { 35, 65 };            // 140...260us valid range for short on pulses
const unsigned int vn1tx_long_range[2]      = { 80, 120 };           // 320...480us valid range for long on pulses
const unsigned int vn1tx_off_thresholds[3]	= { 40, 100, 175 };		 // valid range of off periods for PWM, 160 to 700us (middle value unused)

const unsigned int wh2b_short_range[2]      = { 110, 165 };         // 440...660us
const unsigned int wh2b_long_range[2]       = { 350, 450 };         // 1400...1800us
const unsigned int wh2b_thresholds[3]       = { 110, 260, 450 };    // 440, 1040, 1800us
const unsigned int wh2b_off_thresholds[3]	= { 215, 350, 300 };    // 860, 1000, 1200us (middle value not used)

//
// these signals are symetrical -- identical on & off periods so only one set of thresholds is needed
//
const unsigned int wxsensor_thresholds[3] = { 50, 94, 150 };        // 200, 376, 600us

#if WX_SHIELD_VERSION == 2
//
// flag used to trigger capture of RSSI value during first transition after the preamble,
// and obtain the RSSI value measured during that process. this is really a little state 
// machine that goes from 2=trigger to 1=measure to 0=done.
//
static byte capture_rssi;

static unsigned int rssi; // analog rssi value read from the ADC in the Atmel processor.

#endif

//
// Overflow interrupt routine for timer 2
// When the last bit of a message has been received by the event capture ISR, the state machine will just 
// sit there waiting for the next RF transition -- which won't occur until a new message begins,
// or the receiver's AGC adjusts to the lack of signal and starts detecting noise.
// This timer is designed to expire shortly after the last message bit, so the new message
// can be signalled to background software. This will then happen much sooner than if we waited
// for a new message or noise to begin arriving.
//
ISR(TIMER2_OVF_vect)
{
    byte mask = TIMSK2;
    TIMSK2 =  PULSE_TIMEOUT_DISABLE; // disable further interrupts
    //
    // just for safety, make sure that timer 2 overflow interrupts are enabled before proceeding
    //
    if (!(mask & PULSE_TIMEOUT_ENABLE)) return;

    boolean rx_state_man = (rx_state == RX_STATE_MANCHESTER);

    byte min_cnt = (protocol_version == PROTOCOL_OS1) ? 24 : 40;

    boolean rx_state_ar = 
		(rx_state == RX_STATE_RECEIVING_PSM) || (rx_state == RX_STATE_RECEIVING_PWM);

    if (rx_state_ar) min_cnt = 24;

    if ((rx_state_man || rx_state_ar) && (bufptr.value > min_cnt))

    {
        rx_state = RX_STATE_PACKET_RECEIVED;
    }
    else
    {
        UNSAFE_WEATHER_RESET();
    }	
}
//
// Overflow interrupt vector
// this is need to keep track of overflow events on timer 1 which allows detection
// of very long time periods  while looking for preambles when there might be a lot
// of time between data transitions (or not, depending on how much noise is being detected).
//
ISR(TIMER1_OVF_vect)
{
    timer1_ovfl_count++;
}

void led_update_check()
{
    // most of the time, the LED will be off and we'll return immediately.
    //
    if (LOCK_LED_STATE)
    {
        noInterrupts();
        unsigned int time_expired;
        unsigned int current_time = ICR1;
        //
        // computing the period with an int (16 bits), we can only go about 1/4 of a second before 
        // overflowing. however, the client is expected to call us much more often than that
        // from the run loop. worst thing that happens if not is that the led will stay on 
        // for a longer time period.
        //
        if (current_time < lock_led_on_time)
        {
            // do the math using signed 32-bit integers, but convert the result back to a 16-bit integer
            time_expired = (unsigned int)((long)current_time + 0x10000L - (long)lock_led_on_time);    
        }
        else
        {
            time_expired = current_time - lock_led_on_time;
        }

        if (time_expired > 25000U) // 100msec on time
        {
            LOCK_LED_OFF;
        }
        interrupts();
    }
}

//
// Event capture interrupt routine for timer 1
//
// This is fired every time the received RF signal goes on or off. Since every 
// transition (up or down) is followed by an opposite transition (down or up), the edge detection
// bit for the timer is flipped with every transition.
//
// During signal preambles, we try to figure out what kind of sensor we have by 
// looking at signal on/off times and counting preamble pulses. No attempt is made during
// this time to decode any data bits.
//
// Decoding of the Manchester-coded ASK is performed in this ISR. The technique works by
// examining the time between adjacent RF transitions. This is easy to do using the Atmel
// processor's "timer 1" which has an input specifically designed to detect the timing of transitions
// on input bit 0 of port B. This input even has a de-glitching feature which helps to filter
// out noise on the RF transitions.
//
// It may take a little thinking, but you can prove to yourself these facts about normal
// Manchester-coded signals:
// 
// 1) If the next message bit is the same as the previous message bit, there will need to 
//    be an RF transition (on-to-off or vice versa) in the middle of the bit period.
// 2) If the next message bit is the opposite of the previous bit, there will be no RF transition
//    in the middle of the bit period.
// 3) The time between transitions is either short (1/2 bit period) or long (a full bit period).
// 4) When short transitions occur, they always occur in pairs.
// 
// The ISR below uses these ideas to decode the signal, with the additional knowledge that all 
// bits in the preamble of short transitions are "1" bits. When a long transition occurs, it 
// represents a bit that is the opposite of the preceeding bit. When two short transitions occur,
// it represents a bit that is identical to the preceeding bit.
//
// Special considerations are required for OS protocol version 2.1 messages. 
// See the RF protocol definition document for details.
//
// In fact, RF transmissions from OS units do not have a 50% duty cycle so the definition of
// short and long bit periods are a bit skewed (see protocol document).
//
ISR(TIMER1_CAPT_vect)
{ 
    // do the time-sensitive things first
    TIMSK2 = PULSE_TIMEOUT_DISABLE; // disable timeout events from timer 2
    TCNT2 = 64U; // reset the counter

    unsigned int ovfl = timer1_ovfl_count;
    timer1_ovfl_count = 0;
    //
    // grab the event time
    //
    unsigned int captured_time = ICR1;
    //
    // depending on which edge (rising/falling) caused this interrupt, setup to receive the opposite
    // edge (falling/rising) as the next event.
    //
    boolean rf_was_on = INPUT_CAPTURE_IS_FALLING_EDGE();

    if(!rf_was_on)
    {
        SET_INPUT_CAPTURE_FALLING_EDGE();
        //digitalWrite(DEBUG_FLAG_1_PIN, HIGH);
    }
    else 
    {
        SET_INPUT_CAPTURE_RISING_EDGE();
        //digitalWrite(DEBUG_FLAG_1_PIN, LOW);
    }
    //
    // detect and deal with timer overflows. the timer will overflow about once every
    // 0.26 seconds so it is not all that rare of an occurance. as long as there has only been 
    // one overflow this will work. If there is more than one overflow, just set the period to maximum.
    // there IS a race condition where this logic will sometimes fail but the window is very small.
    // here's the situation: an input edge happens about the same time as timer1 overflows,
    // and the timer gets latched before the overflow, but the timer overflow ISR runs before the
    // edge capture ISR (don't know if this is really possible). In this case, the timer will not
    // have overflowed (it will be equal to 0xFFFF) but the overflow counter WILL indicate an 
    // overflow. Again, the window to create this problem is so narrow this should rarely happen.
    // the worst thing that will happen is that we'll occasionally loose a preamble bit or maybe
    // even a whole message (even more rare). It's certainly better than ignoring overflows altogether.
    //
    // the other thing that can happen if there is no activity on the DATA line for 4.7 hours, is that
    // the overflow counter itself will overflow. again, this might cause us to miss one message at the 
    // worst case, so this is an acceptable risk. In real life, there is always noise on the DATA line
    // if no signal is present which makes this event even more unlikely.
    //
    unsigned int captured_period;
    unsigned int v1_sync_end_period;
    unsigned int elapsed;

    if (ovfl > 1)
    {
        captured_period = 0xFFFFL;
    }
    else
    {
        if (ovfl == 1)
        {
            if (captured_time < previous_captured_time)
            {
                // do the math using unsigned 32-bit integers, but convert the result back to an unsigned 16-bit integer
                captured_period = (unsigned int)((unsigned long)captured_time + 0x10000UL - (unsigned long)previous_captured_time);    
            }
            else
            {
                captured_period = 0xffffL;
            }
        }
        else
        {
            if (captured_time < previous_captured_time)
            {
                // we should never see this...but just in case
                captured_period = 0xffffL;
            }
            else
            {
                captured_period = captured_time - previous_captured_time;
            }
        }
        
    }

    boolean short_period = false;
    boolean long_period = false;
    boolean gotbit;
    boolean valid_period;

    boolean v1_period_okay = false;
    boolean message_bad = false;
    byte adc_high, adc_low;

    //
    // handle RSSI acquisition if we're in one of the data reception states
    //
#if WX_SHIELD_VERSION == 2
    if ((rx_state >= RX_STATE_RECEIVING_PSM) && (rx_state <= RX_STATE_MANCHESTER))
    {
        //
        // read the ADC to get RSSI value if necessary
        //
        if (capture_rssi > 1)
        {
            if (!rf_was_on)
            {
                // this is the beginning of an RF pulse, so enable RSSIC
                // by changing PORTC bit 1 to an input.
                DDRC &= ~0x02;
                capture_rssi = 1;
            }
        }
        else if (capture_rssi)
        {
            if (rf_was_on)
            {
                // this is the end of the RF pulse we used to measure RSSI.
                // disable RSSIC by changing PORTC bit 1 to an output (which has
                // already been programmed to the LOW state).
                //
                DDRC |= 0x02;
                //
                // start an ADC at the end of the rf pulse. if we try to start it at the beginning,
                // the RSSIOUT voltage will not yet reflect the pulse amplitude so we wait until the
                // end. 
                //
                adc_sra = ADCSRA & ~0x10; // get current value and turn off the interrupt status bit
                ADCSRA = adc_sra | 0x10;  // clear the conversion complete bit
                ADCSRA = adc_sra | 0x40;  // write a 1 to the start conversion bit
                ADCSRA = adc_sra;         // set the start bit back to zero (is this necessary?)
            }
            else
            {
                adc_sra = ADCSRA;
                if (adc_sra & 0x10)
                {
                    adc_low = ADCL;
                    adc_high = ADCH;
                    rssi = (adc_high << 8) | adc_low;
                }
                else
                {
                    // the conversion did not finish so set RSSI to zero to flag the problem
                    //
                    adc_low = 0;
                    adc_high = 0;
                    rssi = 0;
                }
                capture_rssi = 0;
            }
        }
    }
#endif

    //
    //===============================//
    //
    //    HUMONGOUS CASE STATEMENT
    //    This implements the receive
    //    state machine.
    //
    //===============================//
    //
    switch (rx_state)
    {
        //
        //===============================//
        //
        //           IDLE STATE
        //
        //===============================//
        //
    case RX_STATE_IDLE:

        bufptr.value = 0;
		protocol_version = 0;
		pulse_count = 0;
		pulse_count = 0;
		might_be_vn1tx = false;
        might_be_f007th = false;

        if (rf_was_on)
        {

            // check to see if the pulse fell into one of 3 valid ranges

            if ((captured_period >= idle_short_range[0]) && (captured_period <= idle_short_range[1]))
            {
                rx_state = RX_STATE_SHORT_PREAMBLE;
                pulse_count = 1;
                break;
            }

            if ((captured_period >= idle_os2_range[0]) && (captured_period <= idle_os2_range[1]))
            {
                rx_state = RX_STATE_V2_PREAMBLE;
                pulse_count = 1;
                break;
            }

            if ((captured_period >= idle_os1_range[0]) && (captured_period <= idle_os1_range[1]))
            {
                rx_state = RX_STATE_V1_PREAMBLE;
                pulse_count = 1;
                break;
            }

            break;
        }
        //
        // RF was OFF
        //

        // two cases here, all having to do with long off periods associated with 
        // PSM signal preamble/sync ... 
        // The version 1 and Maxim receiver shields don't have problems with long off periods in PSM signals,
        // but the v2 shield will show a shortened off pulse off at least 2.5ms during the first
        // long off period, then it takes about 20ms before the data slicer has adjusted
        // to properly decode the bits.

        if ((captured_period >= idle_psm_off_range[0]) && (captured_period < idle_psm_off_range[1]))
        {
            // with version 2 shields we may have a PSM signal here.
            // the next on period will be too long, followed by a correct off period,
            // then a noisy pulse with very short periods at the end. then good behavior.
            // we ignore all transitions for 20ms in this case and then wait for a 2nd long off period.
            rx_state = RX_STATE_PSM_STABILIZE;
            break;
        }
        //
        // for version 1 and 3 shields, even if the first sync pulse is garbled, it will be followed
        // by the proper length off period. we'll see that here.
        //
        // check for PSM signal's very long off period..when detected no info, so just switch to new state
        //
        if ((captured_period >= psm_sync_off_thresholds[0]) && (captured_period < psm_sync_off_thresholds[1]))
        {
            rx_state = RX_STATE_RECEIVING_PSM;
            protocol_version = PROTOCOL_PSM;
            on_thresholds = psm_on_thresholds;
            off_thresholds = psm_off_thresholds;
            break;
        }
        //
        // check the second range of psm sync off periods
        //
        if ((captured_period >= psm_sync_off_thresholds[1]) && (captured_period < psm_sync_off_thresholds[2]))
        {
            rx_state = RX_STATE_RECEIVING_PSM;
            protocol_version = PROTOCOL_PSM;
            on_thresholds = psm_on_thresholds;
            off_thresholds = psm_off_thresholds;
            break;
        }

        UNSAFE_WEATHER_RESET();
        break;
        //
        //===============================//
        //
        //     OS V2.1 PREAMBLE STATE
        //
        //===============================//
        //
    case RX_STATE_V2_PREAMBLE:

        if (rf_was_on)
        {
            if ((captured_period >= rf_on_thresholds[1]) && (captured_period <= rf_on_thresholds[2]))
            {
                pulse_count++;
                if (pulse_count > 20)
                {
                    UNSAFE_WEATHER_RESET();
                }
            }
            else
            {
                UNSAFE_WEATHER_RESET();
            }
            break;
        }
        //
        // RF was OFF
        //
        if ((captured_period > rf_off_thresholds[1]) && (captured_period <= rf_off_thresholds[2]))
        {
            // nothing to do?
            // should we check for a PSM-length off period here...?
            break;
        }
        //
        // check for a short off period
        //
        if ((captured_period > rf_off_thresholds[0]) && 
            (captured_period <= rf_off_thresholds[1]) &&
            (pulse_count >= 14 && pulse_count <= 18))
        {
            //
            // this is the end of a valid version 2.1 message preamble.
            //
            protocol_version = PROTOCOL_OS2;
            RX_LED_ON;
            LOCK_LED_OFF;
            rx_state = RX_STATE_MANCHESTER;
            half_clock_count = 1;
            on_thresholds = rf_on_thresholds;
            off_thresholds = rf_off_thresholds;
            sync_range = 0;
            // 
            // record the first bit
            //
            bufptr.value = 0;
            packet[0] = rf_was_on ? 1 : 0;
            //
            // read ADCH to clear the ADC registers because the timer event
            // will be the end of a short RF pulse and we'll use that to measure RSSI
            //
#if WX_SHIELD_VERSION == 2
            adc_high = ADCH;
            capture_rssi = 2;
#endif          
            break;
        }

        UNSAFE_WEATHER_RESET();
        break;
        //
        //===============================//
        //
        //      SHORT PREAMBLE STATE
        //
        //===============================//
        //
    case RX_STATE_SHORT_PREAMBLE:
        //
        // most common stuff here will be short on and off periods
        // first logic split is whether RF was on or off
        //
        // ========================= RF WAS ON ===============================
        //
        if (rf_was_on)
        {	
			might_be_vn1tx = false;

            if ((captured_period >= idle_short_range[0]) && (captured_period <= idle_short_range[1]))
            {
                pulse_count++;
                if (pulse_count > 28)
                {
                    UNSAFE_WEATHER_RESET();
                }
				//
				// first data pulse (a long "1") after the preamble is hard to distinguish from the
				// preamble pulses with loose tolerances, but can be id'd because it is more than 
				// 100us shorter than the preamble pulses. however, we'd be jumping the gun if we went right into
                // PWM reception state here because things are not all that different in time periods. Instead,
                // we just note this fact in "might_be_vn1tx" and then close the deal if the next off period
                // falls within a rather tight-limited short off interval range for vn1tx.
				//
				might_be_vn1tx = (pulse_count >= 2) && (pulse_count <= 6) && ((captured_period + 25) < previous_on_period);
                break;
            }
            //
            // the only thing left is the WH2B which we detect by the appearance of a long on pulse
            //
            if ((captured_period >= wh2b_long_range[0]) && (captured_period <= wh2b_long_range[1]))
            {
                if ((pulse_count >= 6) && (pulse_count <= 10))
                {
                    rx_state = RX_STATE_RECEIVING_PWM;
                    protocol_version = PROTOCOL_WH2B;
                    LOCK_LED_OFF;
                    on_thresholds = wh2b_thresholds;
                    off_thresholds = wh2b_off_thresholds;
                    sync_range = 0;
                    want_agc_frozen = false;

                    bufptr.value = 1;
                    packet[0] = (captured_period > wh2b_thresholds[1]) ? 8 : 0;
#if WX_SHIELD_VERSION == 2
                    adc_high = ADCH;
                    capture_rssi = 2;
#endif
                    //...other inits?
                }
                else
                {
                    UNSAFE_WEATHER_RESET();
                }
                break;
            }
            //
            // we might be in luck here with the Ambient Weather F007TH unit
            // I think we'll get a standard long rf on pulse here instead of the longer one associated 
            // with WH2C...and this transition marks the end of the preamble as well.
            // 
            if (might_be_f007th &&
                (captured_period >= rf_on_thresholds[1] && 
                captured_period <= rf_on_thresholds[2]) && 
                (pulse_count >= 4) && (pulse_count <= 8))
            {
                rx_state = RX_STATE_MANCHESTER;
                protocol_version = PROTOCOL_F007TH;
                LOCK_LED_OFF;
                half_clock_count = 0; // because long_period is true
                on_thresholds = rf_on_thresholds;
                off_thresholds = rf_off_thresholds;
                sync_range = 0;
                half_clock_count = 0;   // we are at a bit-defining transition
                packet[0] = 0;          // no bits to record...next bit is first one after preamble
                //
                // read ADCH to clear the ADC registers because the timer event
                // will be the end of a long RF pulse and we'll use that to measure RSSI
                //
#if WX_SHIELD_VERSION == 2
                adc_high = ADCH;
                capture_rssi = 2;
#endif
                break;
            }

            UNSAFE_WEATHER_RESET();
            break;
        }

        //
        //===================================== RF WAS OFF =====================================
        //

        might_be_f007th = false;

        if ((captured_period >= idle_short_range[0]) && (captured_period <= idle_short_range[1]))
        {
            break; // nothing else to do...just a valid off period
        }
        //
        // check for OS3 long off period...but WH2B has an off period that fits in this range
		// so we must gate it by the short count being at least greater than 8 or so. if that test fails
        // don't do a reset because WH2B detection is done during the next on-pulse.
        //
        // Ambient Weather F007TH is also going to show a long off period here but the pulse count would 
        // be only 6 (ideally). how do we separate out THAT one? F007TH preamble is 11111101
        // so...6 short pulses, a long off period, then long on period...
		//
        if ((captured_period >= rf_off_thresholds[1]) && (captured_period <= rf_off_thresholds[2]))
        {
            if (pulse_count >= 15)
            {
                //
                // this is the end of a valid version 3.0 message preamble.
                //
                RX_LED_ON;
                LOCK_LED_OFF;
                rx_state = RX_STATE_MANCHESTER;
                protocol_version = PROTOCOL_OS3;
                half_clock_count = 0; // because long_period is true
                on_thresholds = rf_on_thresholds;
                off_thresholds = rf_off_thresholds;
                sync_range = 0;
                want_agc_frozen = false;
                //
                // record the first bit
                //
                bufptr.value = 1;
                packet[0] = rf_was_on ? 1 : 0;
                //
                // read ADCH to clear the ADC registers because the timer event
                // will be the end of a long RF pulse and we'll use that to measure RSSI
                //
#if WX_SHIELD_VERSION == 2
                adc_high = ADCH;
                capture_rssi = 2;
#endif
                // what else?
            }
            else
            {
                // it also might be WH2C...but the length of the next on pulse should help sort that out
                might_be_f007th = true;
            }
            break;
        }
		//
        // the final stage of VN1TX detection is the first short period following the first long data pulse.
        // This signal always begins after preamble with a long pulse (about 150us shorter than preamble pulses)
        // followed by a short off period. The appearance long data pulse was recorded in "maybe_vn1tx".
        //
		if ( might_be_vn1tx && 
			(captured_period >= vn1tx_short_range[0]) && 
			(captured_period <= vn1tx_short_range[1]) )
		{
			protocol_version = PROTOCOL_VN1TX;
			rx_state = RX_STATE_RECEIVING_PWM;
			on_thresholds = vn1tx_thresholds;
			off_thresholds = vn1tx_off_thresholds;
			sync_range = vn1tx_sync_on_range;
			want_agc_frozen = false;

			bufptr.value = 1;
			packet[0] = 8; // first bit is always a "1" (the long data on pulse recorded by maybe_vn1tx).
#if WX_SHIELD_VERSION == 2
			adc_high = ADCH;
			capture_rssi = 2;
#endif
            break;
        }
		//
		// check for the WxSensor's shorter off period
        //
		if ((captured_period >= wxsensor_thresholds[0]) && 
			(captured_period <= wxsensor_thresholds[1])     )
		{
			//
			// there should be 8 short pulses in this preamble
			//
			if ( (pulse_count >= 5) && (pulse_count <= 10) )
			{
				rx_state = RX_STATE_MANCHESTER;
				protocol_version = PROTOCOL_WXSENSOR;
				RX_LED_ON;
				LOCK_LED_OFF;
				half_clock_count = 1;
				bufptr.value = 1;
				packet[0] = 1; // ??? rf_was_on ? 1 : 0;
				on_thresholds = wxsensor_thresholds;
				off_thresholds = wxsensor_thresholds;
				sync_range = 0;
				want_agc_frozen = false;
#if WX_SHIELD_VERSION == 2
				adc_high = ADCH;
				capture_rssi = 2;
#endif
			}
			else
			{
				UNSAFE_WEATHER_RESET();
			}
            break;
		}
		//
		// check for PSM signal's very long off period..when detected no info, so just switch to new state
        // there are two different time ranges for this which could be used to discriminate between sensor types
        // but that is not done here as of now.
		//
		if ((captured_period >= psm_sync_off_thresholds[0]) && (captured_period <= psm_sync_off_thresholds[2]))
		{
			rx_state = RX_STATE_RECEIVING_PSM;
			protocol_version = PROTOCOL_PSM;
			on_thresholds = psm_on_thresholds;
			off_thresholds = psm_off_thresholds;
			//
			// if we stuff the first two bits of the message into the 
			// 2 LSBs of the first nibble, then all of the data bits
			// will be properly aligned remainder of the buffer.
			// this will require that we fill in two zeros to buffer
			// when a long period is detected to create proper alignment for 
			// subsequent repetitions of the message.
			//
			packet[0] = 0;
			bufptr.value = 2; // ready to receive first bit...
#if WX_SHIELD_VERSION == 2
			adc_high = ADCH;
			capture_rssi = 2;
#endif
            break;
		}
        //
        // finally, if nothing else panned out and the captured period is within a range from
        // 2500us to 8000us it might be the off period associated with the long PSM sync off
        // period, but V2 receivers sometimes show a shortened off period followed by a longer
        // on period, such that the total time from pulse off to next pulse off is a valid 
        // psm sync off interval. In that case we go to the stabilize state to check the combined time
        // on the next pulse.
        //
		if ((captured_period >= idle_psm_off_range[0]) && (captured_period < idle_psm_off_range[1]))
		{
			rx_state = RX_STATE_PSM_STABILIZE;
		}
		else
		{
			UNSAFE_WEATHER_RESET();
		}

        break;
        //
        //===============================//
        //
        //       PSM STABILIZE STATE
        //
        //===============================//
        //
    case RX_STATE_PSM_STABILIZE:
        //
        // identical to the PSM sync state, but this is the first repetition of the message,
        // so we need to initialize a bunch of stuff if the total time for the last two transitions is correct.
        //
        if (rf_was_on)
        {
			unsigned int t = captured_period + previous_captured_period;

            if ((t >= psm_sync_off_thresholds[0]) && ( t <= psm_sync_off_thresholds[2])) //elapsed >= psm_slicer_delay)
            {
                rx_state = RX_STATE_RECEIVING_PSM;
                protocol_version = PROTOCOL_PSM;
                on_thresholds = psm_on_thresholds;
                off_thresholds = psm_off_thresholds;
                //
				// if we stuff the first two bits of the message into the 
				// 2 LSBs of the first nibble, then all of the data bits
				// will be properly aligned remainder of the buffer.
				// this will require that we fill in two zeros to buffer
				// when a long period is detected to create proper alignment for 
				// subsequent repetitions of the message.
				//
				packet[0] = 0;
				bufptr.value = 2; // ready to receive first bit...
#if WX_SHIELD_VERSION == 2
                adc_high = ADCH;
                capture_rssi = 2;
#endif
            }
			else
			{
				UNSAFE_WEATHER_RESET();
			}
        }
		else
		{
			UNSAFE_WEATHER_RESET();
		}
        break;
        //
        //===============================//
        //
        //       PSM SYNC STATE
        //
        //===============================//
        //
    case RX_STATE_PSM_SYNC:
        //
        // check that total time for the last two transitions is equal to the 
        // valid range for PSM sync intervals. this works around problems in the V2 receiver
        // where pulses are not properly represented, but total time between RF off edges is correct.
        // this state should only get executed when an RF off edge occurs...else it is an error.
        //
        if (rf_was_on)
        {
			unsigned int t = captured_period + previous_captured_period;

            if ((t >= psm_sync_off_thresholds[0]) && ( t <= psm_sync_off_thresholds[2])) //elapsed >= psm_slicer_delay)
            {
                rx_state = RX_STATE_RECEIVING_PSM;
            }
			else
			{
				if (bufptr.value > 23)
				{
					rx_state = RX_STATE_PACKET_RECEIVED;				
					RX_LED_OFF;
				}
				else
				{
					RX_LED_OFF;
					LOCK_LED_ON;
					lock_led_on_time = captured_time;
					UNSAFE_WEATHER_RESET();
				}
			}
        }
		else
		{
			UNSAFE_WEATHER_RESET();
		}
        break;
        //
        //===============================//
        //
        //       PSM RECEIVING STATE
        //
        //===============================//
        //
    case RX_STATE_RECEIVING_PSM:
        //
        // the message we get here can be repeated up to 9 or so times.
        // so what to do with all those repetitions??? if the msg buffer
        // length was increased to say 240 or so, we could save them all...
        // and do sort of a majority vote at the end as a kind of checksum...?
        // 
        // decode is simple...we get a '0' bit at the end of a short off period,
        // and a '1' bit at the end of a long off period.
        //		
        message_bad = false;     

        if (rf_was_on)
        {
			if ((captured_period >= psm_on_thresholds[0]) && (captured_period < psm_on_thresholds[1]))
			{
				// nothing to do...just a valid on period, no information here
			}
			else
			{
					message_bad = true;
			}
        }
        else
        {
            // three valid cases: short, long or sync-off periods

            gotbit = false;
            valid_period = false;

            if ((captured_period >= psm_off_thresholds[0]) && (captured_period < psm_off_thresholds[1]))
            {
                valid_period = true;
                packet[bufptr.portion.nibble] &= ~(1 << (3 -bufptr.portion.bit));
                gotbit = true;
            }
            else
            {
                if ((captured_period >= psm_off_thresholds[1]) && (captured_period <= psm_off_thresholds[2]))
                {
					valid_period = true;
                    packet[bufptr.portion.nibble] |= 1 << (3 - bufptr.portion.bit);
                    gotbit = true;
                }
            }

            if (gotbit)
            {
				// because PSM has very little preamble, we wait until we have a minimum number
				// of valid bits before turning on the RX led.
				//
                bufptr.value++;
				
				if (bufptr.value == 10)
				{
					RX_LED_ON;
					LOCK_LED_OFF;
				}

                if (bufptr.portion.nibble >= (MAX_MSG_LEN - 1))
                {
                    message_bad = true;
                }
            }
			
            if (!valid_period)
            {
                if (captured_period >= idle_psm_off_range[0] && captured_period <= psm_sync_off_thresholds[2])
                {
                    // fill the current nibble in the output buffer with zeros
                    // if all went well, it should require two zeros to be stuffed in there
                    //
                    while (bufptr.portion.bit)
                    {
                        packet[bufptr.portion.nibble] &= ~(1 << (3 -bufptr.portion.bit));
                        bufptr.value++;
                    }
                    //
                    // init the next nibble  to zero, and skip over the first two bits
                    // this will properly align the next copy of the temperature message,
                    // same as the previous one.
                    //
                    packet[bufptr.portion.nibble] = 0;
                    bufptr.value += 2;

                    rx_state = RX_STATE_PSM_SYNC;
                }
                else
                {
                    message_bad = true;
                }
            }

        }

        if (message_bad)
        {
            if (bufptr.value > 23)
            {
                rx_state = RX_STATE_PACKET_RECEIVED;				
                RX_LED_OFF;
            }
            else
            {
                RX_LED_OFF;
				if (bufptr.value > 10)
				{
					LOCK_LED_ON;
					lock_led_on_time = captured_time;
				}
                UNSAFE_WEATHER_RESET();
            }
        }
        break;

        //
        //===============================//
        //
        //       PWM RECEIVING STATE
        //
        //===============================//
        //
    case RX_STATE_RECEIVING_PWM:
        //
        // the message we get here can be repeated up to 3 or so times.
        // just store them all in the buffer...
        // 
        // decode is simple...we get a '0' bit at the end of a short off period,
        // and a '1' bit at the end of a long off period.
        //
        //want_agc_frozen = false;
        message_bad = false;

        if (rf_was_on)
        {
            // three valid cases: short, long or sync-off periods

            gotbit = false;

            if ((captured_period >= on_thresholds[0]) && (captured_period < on_thresholds[1]))
            {
                packet[bufptr.portion.nibble] &= ~(1 << (3 -bufptr.portion.bit));
                gotbit = true;
            }
            else
            {
                if ((captured_period >= on_thresholds[1]) && (captured_period <= on_thresholds[2]))
                {
                    packet[bufptr.portion.nibble] |= 1 << (3 - bufptr.portion.bit);
                    gotbit = true;
                }
                else
                {
                    //
                    // check for a sync period that would separate the message copies...we 
                    // silently ignore these...but when we get one, we back the buffer pointer up one
                    // spot because there's always an extra bit at the end.
                    //
                    // sync_range pointer may be set to zero in which case we don't recognize any sync periods.
                    //
                    if (sync_range && (captured_period >= sync_range[0]) && (captured_period <= sync_range[1]))
                    {
                        // fill in the rest of the nibble if necessary
                        //
                        while (bufptr.portion.bit)
                        {
                            packet[bufptr.portion.nibble] &= ~(1 << (3 -bufptr.portion.bit));
                            bufptr.value++;
                        }
                        //
                        // init the next nibble  to zero
                        //
                        packet[bufptr.portion.nibble] = 0;
                    }
                    else
                    {
                        message_bad = true;
                    }
                }
            }

            if (gotbit)
            {
                bufptr.value++;

				if (bufptr.value == 10)
				{
					RX_LED_ON;
					LOCK_LED_OFF;
				}

                if (bufptr.portion.nibble >= (MAX_MSG_LEN - 1))
                {
                    message_bad = true;
                }
            }
        }
        else
        {
			//
            // check for valid off period but do nothing else. this can include a valid sync-off period
			// because the sync off happens before sync on pulse. we could use this to force a check for
			// a sync-length pulse next time around too...?
            // although the off_thresholds array contains a value to split between long and short off 
            // periods, we don't use it.
			//
			if ((captured_period < off_thresholds[0]) || (captured_period > off_thresholds[2]))
			{
				message_bad = true;
			}
        }


        if (message_bad)
        {
            //digitalWrite(DEBUG_FLAG_1_PIN,LOW);
            if (bufptr.value > 23)
            {
                rx_state = RX_STATE_PACKET_RECEIVED;				
                RX_LED_OFF;
            }
            else
            {
                RX_LED_OFF;
				if (bufptr.value > 10)
				{
					LOCK_LED_ON;
					lock_led_on_time = captured_time;
				}
                UNSAFE_WEATHER_RESET();
            }
        }
        break;

        //
        //===============================//
        //
        //       V1 PREAMBLE STATE
        //
        //===============================//
        //
    case RX_STATE_V1_PREAMBLE:
        //
        // we're looking for the following values of captured_period here:
        // 1. A short off period
        // 2. A short on period
        // 3. The first sync-off interval
        //
        // anything else means a reset
        //
        v1_period_okay = false;

        if (rf_was_on)
        {
            if ((captured_period >= v1_on_thresholds[0]) && (captured_period < v1_on_thresholds[1]))
            {
                pulse_count++;
                v1_period_okay = true;
                if (pulse_count >= V1_MIN_SYNC_COUNT) 
                {
                    //
                    // a majority of the preamble has been successfully received,
                    // so this is a good time to freeze the AGC. This will make sure
                    // it is frozen for both long rf-off periods that are part of the sync 
                    // portion of version 1.0 RF messages.
                    //
                    want_agc_frozen = true;
                }
            }
        }
        else
        {
            if ((captured_period >= v1_off_thresholds[0]) && (captured_period < v1_off_thresholds[1]))
            {
                v1_period_okay = true;
            }
            else
            {
                if ( (captured_period >= v1_sync_off_range[0]) && (captured_period <= v1_sync_off_range[1]))
                {
                    if (pulse_count > V1_MIN_SYNC_COUNT)
                    {
                        // 
                        // this is the end of the long off period following the preamble, 
                        // just prior to the long sync pulse.
                        //
                        v1_period_okay = true;
                        rx_state = RX_STATE_V1_SYNC_PULSE;						
                    }
                }
            }
        }

        if (!v1_period_okay || (pulse_count > V1_MAX_SYNC_COUNT)) 
        {
            UNSAFE_WEATHER_RESET();
        }
        break;
        //
        //===============================//
        //
        //         V1 SYNC PULSE STATE
        //
        //===============================//
        //
    case RX_STATE_V1_SYNC_PULSE:
        // 
        // this should be the end of the long sync pulse
        //
        if (rf_was_on && (captured_period >= v1_sync_on_range[0]) && (captured_period <= v1_sync_on_range[1]))
        {
            v1_sync_end = captured_time;
            rx_state = RX_STATE_V1_SYNC_END;
            RX_LED_ON;			
        }
        else
        {
            UNSAFE_WEATHER_RESET();
        }
        break;
        //
        //===============================//
        //
        //         V1 SYNC END
        //
        //===============================//
        //
    case RX_STATE_V1_SYNC_END:
        //
        // this should be the end of the long off period following the long sync pulse.
        // it can have two legal ranges of values -- a short duration and a long duration
        //
        short_period = long_period = false;

        if (!rf_was_on)		
        {
            short_period = (captured_period >= v1_sync_end_short[0]) && (captured_period <= v1_sync_end_short[1]);
            if (!short_period)
            {
                long_period =
                    ((captured_period >= v1_sync_end_long[0]) && (captured_period <= v1_sync_end_long[1]));
            }
        }

        if (!short_period && !long_period)
        {
            UNSAFE_WEATHER_RESET();
            break;
        }
        //
        // the off period has a valid duration. perform setup for receiving the data payload
        // and transition to the receiving state.
        //
        half_clock_count = short_period ? 1 : 2;
        packet[0] = 0;
        bufptr.value = short_period ? 0 : 1;

        RX_LED_ON;
        LOCK_LED_OFF;
        rx_state = RX_STATE_MANCHESTER;
        protocol_version = PROTOCOL_OS1;
        //
        // read ADCH to clear the ADC registers because the timer event
        // will be the end of a long RF pulse and we'll use that to measure RSSI
        //
#if WX_SHIELD_VERSION == 2
        adc_high = ADCH;
        capture_rssi = 2;
#endif

        break;
        //
        //=========================================//
        //
        //   DECODING FOR MANCHESTER-CODED SIGNALS
        //
        //=========================================//
        //
    case RX_STATE_MANCHESTER:

        short_period = false;
        long_period = false;
        unsigned int *thr = rf_was_on ? on_thresholds : off_thresholds;

        if (captured_period >= thr[0])
        {
            if (captured_period < thr[1])
            {
                short_period = true;
            }
            else
            {
                if (captured_period <= thr[2])
                {
                    long_period = true;
                }
            }
        }


        if (!short_period && !long_period)
        {
            //
            // transition periods outside the valid ranges for long or short periods occur in two
            // situations: (a) a new message has begun before timer2 can produce a timeout to end
            // the first message, and (b) data corruption through interference, RF noise, etc.
            // In both cases, if there are enough bits to form a message, signal a message complete
            // status and let the downstream software try to decode it. Otherwise, perform a reset
            // and wait for another message.
            //
            if ((bufptr.value > 40) || (protocol_version == PROTOCOL_OS1 && bufptr.value > 24))
            {
                rx_state = RX_STATE_PACKET_RECEIVED;				
                RX_LED_OFF;
            }
            else
            {
                RX_LED_OFF;
                LOCK_LED_ON;
                lock_led_on_time = captured_time;
                UNSAFE_WEATHER_RESET();
            }		
            //flip_debug_flag_1();
            break;
        }
        //
        // update our notion of half-time or time counted in half-clock periods.
        //
        half_clock_count += short_period ? 1 : 2;

        if (half_clock_count & 0x01)
        {
            //
            // we are at the boundary between two clock periods. no bit information is
            // available here. however, we can check to make sure there was a transition
            // at the middle of the last clock period (this is an error check only).
            //
            if (long_period)
            {
                //
                // this means there was no transition in the middle of the last clock period.
                // that is illegal for this modulation format.
                //
                RX_LED_OFF;
                LOCK_LED_ON;
                lock_led_on_time = captured_time;
                //digitalWrite(DEBUG_FLAG_1_PIN, LOW);
                UNSAFE_WEATHER_RESET();
            }
            break;
        }
        //
        // we are at the middle of a clock period and a new bit value is available.
        // handle the duplicated bits in version 2.1 messages first
        //
        if (protocol_version == PROTOCOL_OS2)
        {
            //
            // unless we are at the second bit in a bit pair, this bit is not kept.
            //
            dump_bit = (half_clock_count & 0x03) != 0;

            if (!dump_bit && short_period)
            {
                //
                // this means that the last two bits in this bit pair were identical.
                // version 2.1 protocol always sends the two bits in a pair inverted
                // from one another so this is an error.
                //
                RX_LED_OFF;
                LOCK_LED_ON;
                lock_led_on_time = captured_time;
                //digitalWrite(DEBUG_FLAG_1_PIN, LOW);
                UNSAFE_WEATHER_RESET();
            }
        }
        else
        {
            dump_bit = false;
        }
        //
        // detect a bit for all protocol versions now
        //
        if (!dump_bit)
        {
            if(rf_was_on)
                packet[bufptr.portion.nibble] |= 1 << bufptr.portion.bit;          
            else 
                packet[bufptr.portion.nibble] &= ~(1 << bufptr.portion.bit);				

            bufptr.value++;					
        }			

#if ACCEPT_SQUARE_WAVE
        if (bufptr.value >= SQ_MSG_LEN)
        {
            rx_state = RX_STATE_PACKET_RECEIVED;
            RX_LED_OFF;
            //digitalWrite(DEBUG_FLAG_1_PIN, LOW);
        }
#endif
        break;
        //
        //===============================//
        //
        //     PACKET RECEIVED STATE
        //
        //===============================//
        //
    case RX_STATE_PACKET_RECEIVED:		
        //
        // this most often will happen when a new message begins before the background loop has
        // had a chance to read the current message. in this situation, we just let the new message
        // bits go into the bit bucket...
        //
        want_agc_frozen = false; // for safety -- just in case nobody has cleared it yet
        break;
        //
        //===============================//
        //
        //         DEFAULT STATE
        //
        //===============================//
        //
    default:
        UNSAFE_WEATHER_RESET();
        break;
    }
    //
    //===============================//
    //
    //      END OF CASE STATEMENT
    //
    //===============================//
    //

    previous_captured_time = captured_time;
	previous_captured_period = captured_period;
	if (rf_was_on) previous_on_period = captured_period;

    // guard against buffer overflows this could happen if two messages overlap
    // just right -- probably very rare.
    if (bufptr.portion.nibble >= (MAX_MSG_LEN - 2))
    {
        //LOCK_LED_ON;
        //lock_led_on_time = captured_time;
        UNSAFE_WEATHER_RESET();
    }

    if ((rx_state != RX_STATE_IDLE) && (rx_state != RX_STATE_PACKET_RECEIVED))
    {    
        TCCR2B = T2_PRESCALE_X1024;     // always use a 16msec timeout...can we get away with that???
        TCNT2 = 0U;
        TIFR2 = PULSE_TIMEOUT_FLAG;     // clear overflow bit before enabling interrupts
        TIMSK2 = PULSE_TIMEOUT_ENABLE;        		
    }
}

void wxrx_enable(byte enable)
{
    noInterrupts();
    if (enable)
    {
        //
        // should we attempt to clear any pending interrupts first...?
        // yes, this is mandatory or the capability to write SPI data 
        // to the receiver while it is in the middle of a message will not work.
        //
        TIFR1 = 0x20;
        //
        // enable timer 1 interrupts for input capture events only.
        //
        TIMSK1 = (_BV(ICIE1)) | (_BV(TOIE1));
    }
    else
    {
        TIMSK1 = 0;
    }
    interrupts();
}

void wxrx_init()
{
    noInterrupts();

    want_agc_frozen = false;
	previous_captured_period = 0;
	previous_on_period = 0;

    int i;
    // we never write to the high nibble of packet[] elements so clear them all now.
    for (i=0; i<MAX_MSG_LEN; packet[i++] = 0);

    WEATHER_RESET();
    // 
    // configure ports:
    //
    // ports are initialized with all pins as inputs. 
    // when changed to outputs, all bits are defaulted to zero
    // all pullup resistors are disabled by default
    //
    // setup the bits to control the LEDs.
    // turn off the LEDs.
    //
#if ENABLE_LEDS

    pinMode(RECEIVING_LED_PIN, OUTPUT);
    pinMode(LOCKED_LED_PIN, OUTPUT);
    //
    // Flash the leds a bit to let them know we're alive
    //
    for (i=0; i<10; i++)
    {
        RX_LED_ON;
        delay(50);
        RX_LED_OFF;
        LOCK_LED_ON;
        delay(50);
        LOCK_LED_OFF;
        delay(50);
    }
#endif

    //
    // Setup RSSI pins
    //
#if WX_SHIELD_VERSION == 2
    pinMode(RSSIC_PIN, INPUT);
    digitalWrite(RSSIC_PIN, LOW);
    pinMode(RSSIC_PIN, OUTPUT);
    digitalWrite(RSSIC_PIN, LOW);
#endif
    //
    // setup timer 1 to be triggered by transitions on the DATA  line
    // from the 433.92MHz receiver module. Arduino uses timer 1 for PWM
    // so this code will not work at the same time as some PWM applications
    //
    TCCR1A = 0; // Select normal (simple count-up) mode
    //
    // ICNC1 enables the noise canceller for timer event captures
    // CS10/11/12 bits select the timer clock to be the system clock/64
    // for the Duemillinova, this is 16MHz/64 or 250kHz.
    //
    TCCR1B = ( _BV(ICNC1) | _BV(CS11) | _BV(CS10) );
    SET_INPUT_CAPTURE_RISING_EDGE();
    //
    // disable timer 1 interrupts to start with.
    //
    TIMSK1 = 0; // (_BV(ICIE1)) | (_BV(TOIE1));
    // 
    // setup timer 2 to provide a timeout when waiting for receiver DATA transitions
    // timer2 is also used for PWM by Arduino, so this code may not be compatible with
    // some PWM applications.
    //
    TCCR2A = 0; // select normal mode for the timer
    TCCR2B = T2_PRESCALE_X128;  // select clk/128. 8-bit timer will overflow every 2 msec
    TCNT2 = 0;  // clear the timer count
    TIMSK2 = PULSE_TIMEOUT_DISABLE; // interrupts are disabled to start with...

#if WX_SHIELD_VERSION == 2
    //
    // setup the ADC to read channel 0, triggered by timer capture events (RSSI input)
    //
    DIDR0 = 0x01; // disable digital i/o circuitry on ADC0 input pin
    ADMUX=0x00;   // Select AREF (external input) as Vref, right justified data and channel 0 for input
    ADCSRB=0x07;  // Select timer 1 capture event as trigger source (we don't use that anymore, however)
    //
    // enable ADC but do not start any conversions yet. prescaler is 128 (125kHz conversion rate)
    //
    ADCSRA=0x97;  //0xB7; 

#if ENABLE_HDWR_DEBUG_FLAGS
    pinMode(DEBUG_FLAG_1_PIN, OUTPUT);
    pinMode(DEBUG_FLAG_2_PIN, OUTPUT);
    digitalWrite(DEBUG_FLAG_1_PIN, LOW);
    digitalWrite(DEBUG_FLAG_2_PIN, LOW);
#endif

#endif

    interrupts();
}

boolean wxrx_data_available()
{
    return rx_state == RX_STATE_PACKET_RECEIVED;
}

byte get_wxrx_data(byte *buffer, byte length, byte *protocol)
{
    if (rx_state != RX_STATE_PACKET_RECEIVED) return 0;
    *protocol = protocol_version;
    if (bufptr.portion.bit != 0)
    {
        // bump up to an even multiple of 4 bits
        bufptr.value += 4 - bufptr.portion.bit;
    }
    byte cnt = bufptr.portion.nibble;
    if (length < cnt) return 0;
    memcpy(buffer, packet, cnt);
    return cnt;
}

void start_receiving()
{
    if (rx_state != RX_STATE_PACKET_RECEIVED) return;
    WEATHER_RESET();
}

#if WX_SHIELD_VERSION == 2

unsigned int get_current_rssi() { return rssi; }

unsigned int sample_rssi()
{
    adc_sra = ADCSRA & ~0x10;
    ADCSRA = adc_sra | 0x10; // clear conversion complete bit
    ADCSRA = adc_sra | 0x40; // start a conversion
    ADCSRA = adc_sra;		 // start bit back to zero (is this necessary?)

    // wait for conversion complete or a timeout

    unsigned int start_time = ICR1;
    unsigned int stop_time = ICR1 + 60;
    unsigned int ss = 0;

    do
    {
        adc_sra = ADCSRA;
        if (adc_sra & 0x10) 
        {
            byte adc_low = ADCL;
            byte adc_high = ADCH;
            ss = (adc_high << 8) | adc_low;
            break;
        }
    } while (MILLIS_CMP(stop_time, ICR1) > 0);

    return ss;
}

boolean need_agc_frozen()
{
    return want_agc_frozen;
}

void clear_agc_freeze_request()
{
    want_agc_frozen = false;
}

#endif


#if ENABLE_HDWR_DEBUG_FLAGS
void flip_debug_flag_1()
{
    byte state = digitalRead(DEBUG_FLAG_1_PIN);
    digitalWrite(DEBUG_FLAG_1_PIN, state ? BIT_ZERO : BIT_ONE);
}
void flip_debug_flag_2()
{
    byte state = digitalRead(DEBUG_FLAG_2_PIN);
    digitalWrite(DEBUG_FLAG_2_PIN, state ? BIT_ZERO : BIT_ONE);
}
#endif
