//============================================================================
// Weather Shield for Weather Station Data Logger  
// 
// Version 2.1 (updated for Arduino software release 1.0) 
//
//This application is free software; you can redistribute it and/or
//modify it under the terms of the GNU Lesser General Public
//License as published by the Free Software Foundation; either
//version 3 of the License, or (at your option) any later version.
//
//This application is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR When PARTICULAR PURPOSE. See the GNU
//Lesser General Public License for more details.
//
//You should have received a copy of the GNU Lesser General Public
//License along with this library;  if not, see <http://www.gnu.org/licenses/>
//
//=============================================================================
//
// The seeds for this program originally came out of work done by
// brian@lostbyte.com. There have been many, many changes since then,
// but you can see some of the original work at lostbyte.com.
// A lot of changes have been made since then, so it may be difficult
// to find the resemblance.
//
// Below is one of the (close to) original headers from that work.
//

/***
* Arduino Oregon Scientific V3 Sensor receiver v0.3
* Updates: http://www.lostbyte.com/Arduino-OSV3
* Contact: brian@lostbyte.com
* 
* Receives and decodes 433.92MHz signals sent from the Oregon Scientific
* V3 sensors: THGN801 (Temp/Humid), PCR800 (Rain), WGR800 (Wind), UVN800 (UV)
* V2.1 sensors: THGR122NX, THGN123N, UVR128
*
* Special outputs for WSDL (Weather Station Data Logger) program.
*
* For more info: http://lostbyte.com/Arduino-OSV3
*
* Hardware based on the Practical Arduino Weather Station Receiver project
* http://www.practicalarduino.com/projects/weather-station-receiver
* 
* Special thanks to Kayne Richens and his ThermorWeatherRx code.  
* http://github.com/kayno/ThermorWeatherRx
*
*/

#include <avr/pgmspace.h>   // to use PROGMEM macro

#include <WxReceiverConfig.h>

//
// use the defines below to configure which optional sensors are supported by this 
// Arduino sketch. Version 2 shields MUST select the BMP085 sensor.
// Version 1 shields should select BMP085 or BMP180 to reflect the barometer installed on the shield.
//
#define ENABLE_RECEIVER       1
#define ENABLE_SHT1X_SENSOR   1
#define ENABLE_BMP085_SENSOR  1
#define ENABLE_BMP180_SENSOR  0
#define ENABLE_CRC_OUTPUT     0
//
// these macros enable output of weather data over USB, but they only will
// have an effect if the corresponding weather hardware is enabled.
//
#define ENABLE_RECEIVER_PRINT 1
#define ENABLE_SHT1X_PRINT    1
#define ENABLE_BMP085_PRINT   1
//
// set this to 1 to print out averages of the receiver noise level using 
// analog signal strength.
//
#define RX_NOISE_CHECK        1
//
// set this to 1 to output frequent non-averaged barometer readings to see how noisy they are.
//
#define BMP085_NOISE_CHECK    0
//
// for shields using the MC33596 receiver, this turns on code that will read the
// digital value of received signal strength measured by the MC33596 during the 
// RSSI stands for "received signal strength indicator"
// reception of each message. It is necessary to use the digital value combined
// with the analog reading to get the full dynamic range from the RSSI capability in the MC33596.
//
#define ENABLE_DIGITAL_RSSI   1

#if ENABLE_BMP085_SENSOR && ENABLE_BMP180_SENSOR
#error "Both BMP180 and BMP085 sensors have been enabled. This is not supported."
#endif

#if ENABLE_BMP085_SENSOR
#include <ExtendedWire.h>
#include <Bmp085.h>
#define BMP Bmp085
#endif

#if ENABLE_BMP180_SENSOR
#include <ExtendedWire.h>
#include <Bmp180.h>
#define BMP Bmp180
#endif

#if ENABLE_SHT1X_SENSOR
#include <Sensirion.h>
#endif

#if ENABLE_RECEIVER

#if WX_SHIELD_VERSION == 2
#include <WxSpi.h>
#endif

#include <WxReceivers.h>

#endif
//
// these macros will be defined even if the associated item is not enabled
// it would require a whole lot more ifdef's to change that, and
// having them defined if not needed does not really hurt anything.
//
#if WX_SHIELD_VERSION == 2
#define SHT1X_DATA_PIN  7  
#define SHT1X_CLK_PIN   6 
#define BMP085_EOC_PIN 16
#define BMP085_RST_PIN  9
#endif

#if WX_SHIELD_VERSION == 1
#define SHT1X_DATA_PIN  4 
#define SHT1X_CLK_PIN   3 
#define BMP085_EOC_PIN 17
#define BMP085_RST_PIN  9 /* note: BMP085 reset was not connected on this version of the shield board */
#endif

//
// MILLIS_CMP(a,b) returns the sign of (a-b) -- or zero if a and b are identical
// in essence, this does a signed comparison of unsigned long numbers and makes the assumption
// that when two numbers differ by more than mm_diff, that an overflow or underflow must have 
// occurred. the over/underflow is "fixed" and the proper answer is returned
//
#define MILLIS_CMP(a,b) ( (a==b) ? 0 : ( (a>b) ? (((a-b)>mm_diff) ? -1 : 1) : (((b-a)>mm_diff) ? 1 : -1) ) )
const unsigned long mm_diff = 0x7fffffffUL;

#if ENABLE_BMP085_SENSOR || ENABLE_BMP180_SENSOR
//
// these macros determine how barometer readings are taken and averaged prior to being sent out
// over the USB port.
// bmpRdgIntvl is the time in milliseconds between each overall measurement that goes out over USB
// bmpAvgCount is the number of BMP085 readings that are summed to form one piece of USB data
// bmpAvgIntvl is the time in milliseconds between individual BMP085 readings that are summed to form an average
// bmpOssMode  is the internal averaging mode for the BMP085 sensor -- see the Bosch data sheet.
//
// The summed readings are not divided by the averaging count before transmission over USB -- that task
// is left to the host computer's software.
//
#if BMP085_NOISE_CHECK
const unsigned long bmpRdgIntvl =  500UL;	/* twice a second, start an single reading with... */
const unsigned int bmpAvgCount  =     1U;   /* 1 reading */
const unsigned long bmpAvgIntvl =   54UL;	/* this value is not used when the avg count == 1 */
const byte bmpOssMode           =     3U;	/* OSS mode 3 takes the average of 8 readings internal to the BMP085 */
#else
const unsigned long bmpRdgIntvl = 60000UL; /* once a minute, start an averaged reading with... */
const unsigned int bmpAvgCount  = 1024U;   /* 1024 readings spaced... */
const unsigned long bmpAvgIntvl = 54UL;    /* 54 msec apart: it should take about 55.3 seconds for all 1024 readings */
const byte bmpOssMode           = 3U;      /* OSS mode 3 takes the average of 8 readings internal to the BMP085 */
#endif
//
// this defines a small state machine used to manage the BMP085's operation 
//
#define BMP085_STATE_INIT	0		// indicates we need to get cal data 
#define BMP085_STATE_IDLE	1		// waiting for the timer to expire on the next measurement
#define BMP085_STATE_BUSY	2		// in process of taking a reading

static byte bmp085_state;

static unsigned long bmpNextRdg;	// timer value for the next scheduled barometer reading

#endif  // ENABLE_BMP085_SENSOR

#if ENABLE_RECEIVER
//
// some versions of sketches have problems where they hang up every few hours and cease to
// receive RF messages. Until that problem is fixed, this is a watchdog timer used to detect
// the problem. Upon detection, a special USB message is sent to the host computer which amounts
// to a request for rebooting Arduino. The current sketch version uses a newer RF receiver
// library which combines the code used to receive OS 2.1 and 3.0 protocols and for some unknown
// reason the hangs are much, much less frequent now. Perhaps only once every few days or so.
//
#define RF_TIMEOUT   180000UL
static unsigned long rfWatchdog;
//
// There is a simple state machine used to handle acquisition of digital RSSI information
// from the MC33596 after a complete message has been received. These are the states.
//
#define RX_RSSI_IDLE 0
#define RX_RSSI_RQST 1
#define RX_RSSI_ACQ  2

static int rx_rssi_state;
static byte rx_rssi_register; // holds the digital RSSI value read from the MC33596 receiver.

#if ENABLE_RECEIVER_PRINT
static const PROGMEM char protocolHeaders[24][3] = 
{
    {'?','?','?'},
    {'O','S','1'},
    {'O','S','2'},
    {'O','S','3'},
    {'?','?','?'},  // 4
    {'?','?','?'},
    {'?','?','?'},
    {'?','?','?'},
    {'?','?','?'},
    {'?','?','?'},
    {'?','?','?'},
    {'?','?','?'},
    {'?','?','?'},
    {'?','?','?'},
    {'?','?','?'},
    {'?','?','?'},
    {'P','S','M'}, // 16
    {'V','N','1'},
    {'W','H','2'},
    {'0','0','7'},
    {'P','S','F'},
    {'?','?','?'},
    {'?','?','?'},
    {'W','X','R'}
};
#endif

#endif

// used for fast generation of ASCII hex strings
const char hexChars[16] = { 
	'0', '1', '2', '3', '4', '5', '6', '7',
	'8', '9', 'A', 'B', 'C', 'D', 'E', 'F'   
};

char printBuf[160];
byte pbHead = 0;
byte pbTail = 0;
byte pbCnt = 0;

void print2buf(char *s)
{
	int n = strlen(s);
	int space=sizeof(printBuf)-pbCnt;
	if (space < n) return;
	for (; *s; s++)
	{
		printBuf[pbHead] = *s;
		pbHead = (pbHead+1) % sizeof(printBuf);
	}
	pbCnt += n;
}

void print2buf(int i)
{
	char s[10];
	sprintf(s,"%1d",i);
	print2buf(s);
}

void print2buf(char c)
{
	char s[2];
	s[0] = c;
	s[1] = 0;
	print2buf(s);
}

void println2buf(char *s)
{
	print2buf(s);
	print2buf("\n");
}

void println2buf(int i)
{
	print2buf(i);
	print2buf("\n");
}

void outputPrintBuf(int maxChars)
{
	while (maxChars > 0)
	{
		if (pbCnt)
		{
			char c = printBuf[pbTail];
			bool newline = false;
			pbTail = (pbTail + 1) % sizeof(printBuf);
			if (--pbCnt)
			{
				if (printBuf[pbTail] == '\n')
				{
					newline = true;
					pbCnt--;
					pbTail = (pbTail + 1) % sizeof(printBuf);
				}
			}
			if (newline) Serial.println(c); else Serial.print(c);
		}
		maxChars--;
	}
}

//
// prints an unsigned int (16 bits) in hexadecimal format.
//
void print_unsigned_int(unsigned int x, boolean new_line)
{
	char s[5];
	s[0] = hexChars[(x >> 12)];
	s[1] = hexChars[(x >>  8) & 0x0F];
	s[2] = hexChars[(x >>  4) & 0x0F];
	s[3] = hexChars[x & 0x0F];
	s[4] = 0;
	new_line ? println2buf(s) : print2buf(s);
}
//
// prints unsigned long (32 bits) in hexadecimal format
//
void print_unsigned_long(unsigned long x, boolean new_line)
{

	print_unsigned_int((unsigned int)(x >> 16), false);
	print_unsigned_int((unsigned int)(x & 0x0000FFFFUL), new_line);
}
//
// initialization function for the entire Arduino sketch.
// this function is automatically called by the boot loader 
// after the processor has been reset.
//
void setup()
{
	Serial.begin(57600);

#if ENABLE_SHT1X_SENSOR
	Sensirion.init(SHT1X_CLK_PIN, SHT1X_DATA_PIN);
	Sensirion.start_measurement();
#endif

#if ENABLE_RECEIVER
	rx_rssi_state = RX_RSSI_IDLE;
	WxReceivers.init();
	WxReceivers.start();
#endif

#if ENABLE_BMP085_SENSOR 
	bmp085_state = BMP085_STATE_INIT;
	Bmp085.init(BMP085_EOC_PIN, BMP085_RST_PIN);
	Bmp085.start();
#endif


#if ENABLE_BMP180_SENSOR 
	bmp085_state = BMP085_STATE_INIT;
	Bmp180.init();
	Bmp180.start();
#endif

	rfWatchdog = millis() + RF_TIMEOUT;

	println2buf("WXS:5.0.2"); // announce the firmware version
	//
	// announce the weather shield hardware version in use.
	//
#if WX_SHIELD_VERSION == 2
	println2buf("WXH:2.0");
#endif

#if WX_SHIELD_VERSION == 1
	println2buf("WXH:1.0");
#endif

#if ENABLE_RECEIVER
	
	println2buf("OPT:WRX");

#if ENABLE_CRC_OUTPUT
	println2buf("OPT:CRC");
#endif

#if DETECT_PSM
	println2buf("OPT:PSM");
#endif
        
#if DETECT_VN1TX
	println2buf("OPT:VN1");
#endif

#if DETECT_WXSENSOR
	println2buf("OPT:WXR");
#endif

#endif

#if ENABLE_SHT1X_SENSOR
	println2buf("OPT:SHT");
#endif

#if ENABLE_BMP085_SENSOR || ENABLE_BMP180_SENSOR
	println2buf("OPT:BMP");
#endif

#if ENABLE_RECEIVER && RX_NOISE_CHECK && (WX_SHIELD_VERSION == 2)
    for (int k=0; k<RX_NOISE_CHECK; k++) 
	{
        print2buf("RXN:");
        print_unsigned_int(WxReceivers.force_read_rssi(9000), true);
    }
#endif
        
}

#if (WX_SHIELD_VERSION == 2) && ENABLE_RECEIVER // this is the only version that accepts commands over USB right now...
//
// The sketch accepts a limited number of commands from the USB interface. For simplicity,
// each command is a single character as follows:
//
// Characters "A" through "D" set the RF input attenuation as follows:
//
// A = 0dB
// B = 8dB
// C = 16dB
// D = 30dB
//
// The sketch will respond with the message ATN:x,y where "x" is the number of dB attenuation
// requested, and "y" is 1 if the command worked and 0 if the command failed.
//
// RF attenuation is useful if the receiver is being overloaded (unlikely), and for determining
// the signal strength of sensors above about -46dBm.
//
// The "R" command requests an immediate RSSI measurement, not synchronized with any RF data. 
// This is useful for determining the receiver's noise floor, although care should be used as
// the data is taken when requested in time and will measure signal level if there happens to 
// be an RF signal present when the request is made.
//
// The sketch responds to the RSSI request with RSS:x where x is the RSSI value.
//
void process_command(int rqst)
{
	int dB;
	unsigned int rssi;
	boolean setok;
    //
    // our client may use line feeds and/or "at" symbols (@) to flush
	// the serial port, so we just discard these characters when received.
    //
	if (rqst == 0x0A || rqst == '@') return;

	switch (rqst)
	{
	case 'A':
	case 'B':
	case 'C':
	case 'D':
		dB = 30;
		if (rqst < 'D') dB = 8 * (rqst - 'A');
		setok = WxReceivers.set_rf_attenuation(dB);
		print2buf("ATN:");
		print2buf(dB);
		println2buf((char*)(setok ? ",1" : ",0"));
		break;
	case 'R':
		rssi = WxReceivers.force_read_rssi(100);
		print2buf("RSS:");
		print_unsigned_int(rssi, true);
		break;
	default:
		if (rqst != 0) // don't report nulls
		{
			print2buf("ERR:");
			println2buf(rqst);
		}
		break;
	}
}
#endif

//
// This is the main run loop of the sketch, called automatically from the 
// Arduino run loop.
//
void loop() 
{
	byte packet[120];			// a buffer used to hold receive data from the receiver library
	byte rf_protocol_version;	// Set to either 2 or 3, to indicate the OS RF protocol version

	char cPacket[92];			// a buffer used to hold the USB message containing RF message data
	unsigned int msgLen;		// holds the message length returned from the receiver library

#if ENABLE_RECEIVER

#if WX_SHIELD_VERSION == 2
	if (!digitalRead(0)) // check for incoming USB data (a command)
	{
		// The receiver needs to be paused during command processing. We'll lose any RF messages
		// that come in during this time.
		//
		WxReceivers.pause();
		int rqst = Serial.read();
		if (rqst >= 0) process_command(rqst);
		WxReceivers.resume();    
	}
#endif

	WxReceivers.step();		// run the receiver state machine.

#endif // ENABLE_RECEIVER

#if ENABLE_SHT1X_SENSOR
	Sensirion.step();		// run the Sensirion SHT1X state machine.
	//
	// if a measurement is available from the SHT1X sensor, send a message
	// out over USB containing the raw measurement data. The host computer
	// is responsible for converting the readings into calibrated 
	// temperature and humidity values.
	//
	if (Sensirion.dataReady())
	{
		if ( Sensirion.dataValid() )
		{
			unsigned int t,rh;
			Sensirion.getRawData(&t, &rh);

#if ENABLE_SHT1X_PRINT
			print2buf("SHT:");
			print_unsigned_int(t, false);
			print2buf(",");
			print_unsigned_int(rh, true);
#endif
		}
		else
		{
			println2buf("SHT:CRC,CRC");
		}
		outputPrintBuf(99);
		Sensirion.start_measurement(); // start the next measurement
	}
#endif

#if ENABLE_BMP085_SENSOR || ENABLE_BMP180_SENSOR

	// 
	// run the barometer state machine. these repetitive calls are used by the 
	// BMP085 library to do things like acquisition of cal data and generation
	// of averaged readings.
	//
	BMP.step();		
	//
	// this is the state machine to manage the barometer. It only has three states.
	//
	switch (bmp085_state)
	{
	case BMP085_STATE_BUSY:
		//
		// In this state, the BMP085 library is busy taking some number of measurements
		// from the sensor to create an average reading (actually, only the sum of values
		// is computed).
		//
		// if a measurement is available from the barometer, send the raw
		// measurement data out over the USB port. The host computer is 
		// responsible for completing any averaging process since the raw
		// data is only the sum of readings and still needs to be divided by
		// the number of averaging measurements.
		//
		if (BMP.dataAvailable())
		{
			unsigned int navg;
			unsigned long p;
			unsigned long t;
			if (BMP.getRawData(&t, &p, &navg))
			{
#if ENABLE_BMP085_PRINT
				print2buf("BMX:");
				print_unsigned_long(t, false);
				print2buf(",");
				print_unsigned_long(p, true);
#endif
			}
			bmp085_state = BMP085_STATE_IDLE;
		}
		break;

	case BMP085_STATE_IDLE:
		//
		// In this state we've finished an averaged reading sequence and are 
		// waiting for the timer (bmpNextRdg) to expire so that we can begin
		// another averaged reading sequence.
		//
		if ( MILLIS_CMP( millis(), bmpNextRdg) == 1 )
		{
			BMP.startMeasurement();
			bmpNextRdg += bmpRdgIntvl;
			bmp085_state = BMP085_STATE_BUSY;
		}
		break;

	case BMP085_STATE_INIT:
		// 
		// in this sate, the barometer has been initialized and we are waiting 
		// for calibration data constants to be available from the library
		// once they are available, the first averaged measurement can be started.
		//
		if (BMP.calDataAvailable())
		{
#if ENABLE_BMP085_PRINT
			int cal[11];
			if (BMP.getCalData(cal, 11) == 11U)
			{
				print2buf("BMO:");  // OSS mode
				print_unsigned_int(bmpOssMode, true);
				print2buf("BMV:");  // averaging count
				print_unsigned_long(bmpAvgCount, true);

				for (int k=0; k<11; k++)
				{
					print2buf("BM");
					print2buf(hexChars[k]);
					print2buf(":");
					print_unsigned_int((unsigned int)cal[k], true);
					outputPrintBuf(99);
				}
			}
			else
			{
				println2buf("BM#"); // this message indicates that we received invalid cal data
			}
#endif
			BMP.setupAveraging(bmpOssMode, bmpAvgCount, bmpAvgIntvl);
			BMP.startMeasurement();
			bmpNextRdg = millis() + bmpRdgIntvl;
			bmp085_state = BMP085_STATE_BUSY;
		}
		break;
	}

#endif  // ENABLE_BMP085_SENSOR


#if ENABLE_RECEIVER
	//
	// read this only once during the loop to avoid any risk of it changing
	// state inbetween the two accesses to it below.
	// TODO: move this code into the WxReceivers class
	//
	boolean wxrxda = WxReceivers.data_available();
        
	if (wxrxda)
	{
#if WX_SHIELD_VERSION == 2  // process RSSI if supported by receiver hardware
		switch (rx_rssi_state)
		{
		case RX_RSSI_IDLE:
#if ENABLE_DIGITAL_RSSI
			WxReceivers.request_rssi();
			rx_rssi_state = RX_RSSI_RQST;
			//println2buf("WXS:RQST");
#else
			rx_rssi_register = (byte)0x11;
			rx_rssi_state = RX_RSSI_ACQ;
#endif
			break;
		case RX_RSSI_RQST:
			if (WxReceivers.rssi_available())
			{
				rx_rssi_register = WxReceivers.rssi_register_value();
				rx_rssi_state = RX_RSSI_ACQ;
				//println2buf("WXS:ACQ");
			}
			break;
		case RX_RSSI_ACQ:
			break;
		default:
			rx_rssi_state = RX_RSSI_IDLE;
			break;
		}
#endif  // WX_SHIELD_VERSION == 2

	}
	else
	{
		if ( MILLIS_CMP(millis(), rfWatchdog) == 1 )
		{
			print2buf("XXX:");
			println2buf((int)rx_rssi_state);
			//print2buf(",");
			//print2bufWxReceivers.state());
			//print2buf(",");
			//println2buf(WxReceivers.interruptCounter());
			WxReceivers.force_reset();
			rfWatchdog = millis() + RF_TIMEOUT;
		}
	}

#if WX_SHIELD_VERSION == 2
	if(wxrxda && (rx_rssi_state == RX_RSSI_ACQ))
#else
	if (wxrxda)
#endif
	{
		byte msgLen = WxReceivers.get_data(packet, sizeof(packet), &rf_protocol_version);

		// print2buf("WXS:RX"); println2buf((int)msgLen);

		if (msgLen > 0)
		{      
			int k;
            // int start = (rf_protocol_version == 1 || rf_protocol_version > 15) ? 0 : 1;
			int dest = 0;
            
#if ENABLE_RECEIVER_PRINT

            unsigned int hdrOffset = rf_protocol_version * 3;
            if (hdrOffset >= sizeof(protocolHeaders)) hdrOffset = 0;

            for (dest=0; dest<3; dest++)
            {
                cPacket[dest] = pgm_read_byte_near((char*)protocolHeaders + hdrOffset++);
            }

			cPacket[dest++] = ':';

			int msgEnd = msgLen;
            //
            // for no crc output, remove two nibbles from the end of every version 1,2,3 message. 
            // for crc output, remove the two nibbles only for version 1 messages.
            // we do not bother to trim any CRC data from other protocols; let the client deal with it.
            // longer term we should make this more consistent...either always send CRC or never send CRC.
            //
#if ENABLE_CRC_OUTPUT
			if (rf_protocol_version == PROTOCOL_OS1) msgEnd -= 2;
#else
            if (PROTOCOL_IS_OS(rf_protocol_version)) msgEnd -= 2;
#endif

			if (msgEnd > (sizeof(cPacket)-2)) msgEnd = sizeof(cPacket) - 2;

			for (k=0; k < msgEnd; k++)
			{
				cPacket[dest++] = hexChars[packet[k]];
			}

#if WX_SHIELD_VERSION == 2
			unsigned int rssi = WxReceivers.get_rssi();

			cPacket[dest++] = ',';
			cPacket[dest++] = hexChars[(rssi >> 8) & 0x0f];
			cPacket[dest++] = hexChars[(rssi >> 4) & 0x0f];
			cPacket[dest++] = hexChars[rssi & 0x0f];

			cPacket[dest++] = ',';
			cPacket[dest++] = hexChars[(rx_rssi_register >> 4) & 0x0f];
			cPacket[dest++] = hexChars[rx_rssi_register & 0x0f];
#endif

			cPacket[dest++] = 0;
			println2buf(cPacket);

			rfWatchdog = millis() + RF_TIMEOUT;
#endif
		}

#if WX_SHIELD_VERSION == 2
		rx_rssi_state = RX_RSSI_IDLE;
#endif

		WxReceivers.resume_receiving();
	}
#endif

	outputPrintBuf(1);
}

