//============================================================================
//Weather Station Data Logger : Weather Shield for Arduino
//Copyright  2010, Weber Anderson
// 
//This application is free software; you can redistribute it and/or
//modify it under the terms of the GNU Lesser General Public
//License as published by the Free Software Foundation; either
//version 3 of the License, or (at your option) any later version.
//
//This application is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR When PARTICULAR PURPOSE. See the GNU
//Lesser General Public License for more details.
//
//You should have received a copy of the GNU Lesser General Public
//License along with this library; if not, see <http://www.gnu.org/licenses/>
//
//=============================================================================

#include "bmp.h"
#include "Bmp085.h"
// this uses a modified version of the wire library, which allows
// multiple commands without deselecting the slave.
#include "ExtendedWire.h"

//
// MILLIS_CMP(a,b) returns the sign of (a-b) -- or zero if a and b are identical
// in essence, this does a signed comparison of unsigned long numbers and makes the assumption
// that when two numbers differ by more than mm_diff, that an overflow or underflow must have 
// occurred. the over/underflow is "fixed" and the proper answer is returned
//
#define MILLIS_CMP(a,b) ( (a==b) ? 0 : ( (a>b) ? (((a-b)>mm_diff) ? -1 : 1) : (((b-a)>mm_diff) ? 1 : -1) ) )
const unsigned long mm_diff = 0x7fffffffUL;

// create the instance for users to access the class through.

BoschBmp085 Bmp085 = BoschBmp085();

BoschBmp085::BoschBmp085()
{
  cal_data_available = false;
  data_available = false;
  state = STATE_INIT;
  navg = 1U;
  avg_count = 0;
  meas_rqst = false;
  read_failures = 0;
  clock_rate = 400000UL;
  oss_mode = 3;
  oss_rqst = BMP_OSS_MODE(oss_mode);
  cal_data = (int*)malloc(sizeof(int) * CAL_DATA_LENGTH);

  failmsg = 0;
  prev_stat = 0xffU;
}

BoschBmp085::~BoschBmp085()
{
  if (cal_data != 0) free(cal_data);
}

void BoschBmp085::init()
{
  init(DEFAULT_EOC_PIN, DEFAULT_RST_PIN);
}

void BoschBmp085::init(byte BMP085_eoc_pin, byte BMP085_reset_pin)
{
  noInterrupts();
  eoc_pin = BMP085_eoc_pin;
  rst_pin = BMP085_reset_pin;
  
  pinMode(eoc_pin, INPUT);
  pinMode(rst_pin, OUTPUT);

  digitalWrite(rst_pin, 0);
  delayMicroseconds(2);
  digitalWrite(rst_pin, 1);
  interrupts();

  Wire.begin(false);
  Wire.setSpeed(100000UL);
}

void BoschBmp085::start()
{
  state = STATE_INIT;
  // request the chip id code
  Wire.kickoffRequestFromAt( BMP_ADDR, BMP_CHIP_ID, 1U );
}

void BoschBmp085::setupAveraging(byte OSSmode, unsigned int avgCount, unsigned long interval)
{
  if (OSSmode > 3)
    oss_mode = 3;
  else
    oss_mode = OSSmode;

  oss_rqst = BMP_OSS_MODE(oss_mode); // pre-compute the register value to save time

  navg = avgCount;
  avg_interval = interval;
}

boolean BoschBmp085::calDataAvailable()
{
  return cal_data_available;
}

byte BoschBmp085::getCalData(int* buffer, byte length)
{
  if (!cal_data_available) return 0;
  byte count = CAL_DATA_LENGTH;
  if (length < count) return 0;
  memcpy(buffer, cal_data, sizeof(int) * count);
  return count;
}

void BoschBmp085::startMeasurement()
{
  avg_count = 0U;
  temperature = 0UL;
  pressure = 0UL;
  data_available = false;
  meas_rqst = true;
}

boolean BoschBmp085::dataAvailable()
{
  return data_available;
}

boolean BoschBmp085::getRawData(unsigned long* temp, unsigned long* press, unsigned int* avgCount)
{
  if (!data_available) return false;
  *temp = temperature;
  *press = pressure;
  *avgCount = avg_count; // return actual count, not the setting (navg)
  return true;
}


void BoschBmp085::step()
{
  uint8_t stat = Wire.status();

  if (stat != prev_stat)
  {
    //Serial.print("Status: "); Serial.println((int)Wire.status());
    prev_stat = stat;
  }

  if (stat == WIRE_STATUS_BUSY) return;

  uint8_t rc;
  uint8_t cal_offset;
  uint8_t cal_index;

  switch (state)
  {
  case STATE_INIT:
    // we're expecting a read from the chip id register here...
    if ( (Wire.status() != WIRE_STATUS_SUCCESS) || (Wire.available() != 1) )
    {
      // usually caused by the chip being dead -- most likely, it did 
      // not ACK it's address.
      //Serial.print("Fail, avail="); Serial.println((int)Wire.available());
      state = STATE_FAIL;
    }
    else
    {
      // get the chip id
      if (0) //Wire.receive() != BMP_CHIP_ID)
      {
        state = STATE_FAIL;
      }
      else
      {
        // good to go...start reading the EEPROM calibration data
        memset(cal_data, 0, CAL_DATA_LENGTH * sizeof(int));
        cal_data_reg = BMP_CAL_START;
        rc = Wire.kickoffRequestFromAt(BMP_ADDR, cal_data_reg, 1U);
        state = (rc == WIRE_STATUS_SUCCESS) ? STATE_READ_CAL : STATE_FAIL;
        //Serial.println("Read Cal");
      }
    }
    break;

  case STATE_READ_CAL:
    // we're expecting a read from the chip id register here...
    if ( (Wire.status() != WIRE_STATUS_SUCCESS) || (Wire.available() != 1) )
    {
      state = STATE_FAIL;
    }
    else
    {
      cal_offset = cal_data_reg++ - BMP_CAL_START;
      cal_index = cal_offset >> 1;
      cal_data[cal_index] <<= 8;
      cal_data[cal_index] |=  Wire.receive();

      if (cal_data_reg > BMP_CAL_END)
      {
        cal_data_available = 1;
        state = STATE_IDLE;
        //Serial.println("Idle");
      }
      else
      {
        rc = Wire.kickoffRequestFromAt(BMP_ADDR, cal_data_reg, 1U);
        state = (rc == WIRE_STATUS_SUCCESS) ? STATE_READ_CAL : STATE_FAIL;
      }
    }
    break;

  case STATE_IDLE:
    if (!meas_rqst) break; // wait for background loop to request a new msmt
    //
    // start off by requesting a temperature measurement
    //
    Wire.beginTransmissionAt(BMP_ADDR, BMP_MEAS_RQST);
    Wire.send(oss_rqst);
    rc = Wire.kickoffTransmission();
    next_reading = millis() + avg_interval;
    state = (rc == WIRE_STATUS_SUCCESS) ? STATE_MEAS_PRESS : STATE_FAIL;
    //Serial.println("Meas temp");
    break;

  case STATE_MEAS_TEMP:
    if (digitalRead(eoc_pin) != 0)
    {
      tval = 0;
      meas_data_reg = BMP_DATA_MSB;
      // temperature measurement is finished. request the data.
      rc = Wire.kickoffRequestFromAt(BMP_ADDR, meas_data_reg,  1U);
      state = (rc == WIRE_STATUS_SUCCESS) ? STATE_READ_TEMP : STATE_FAIL;
    }
    break;

  case STATE_READ_TEMP:
    if ( (Wire.status() != WIRE_STATUS_SUCCESS) || (Wire.available() != 1) )
    {
      state = STATE_FAIL;
    }
    else
    {
      tval <<= 8;
      tval |= (unsigned long)Wire.receive();

      if (meas_data_reg == BMP_DATA_LSB)
      {
        temperature += tval;

        if (++avg_count >= navg)
        {
          data_available = true;
          meas_rqst = false;
          state = STATE_IDLE;
        }
        else
        {
          state = STATE_WAIT_NEXT;
        }

      }
      else
      {
        rc = Wire.kickoffRequestFromAt(BMP_ADDR, ++meas_data_reg, 1U);
        if (rc != WIRE_STATUS_SUCCESS) state = STATE_FAIL;
      }
    }
    break;

  case STATE_MEAS_PRESS:
    if (digitalRead(eoc_pin) != 0)
    {
      pval = 0;
      meas_data_reg = BMP_DATA_MSB;
      // temperature measurement is finished. request the data.
      rc = Wire.kickoffRequestFromAt(BMP_ADDR, meas_data_reg,  1U);
      state = (rc == WIRE_STATUS_SUCCESS) ? STATE_READ_PRESS : STATE_FAIL;
    }
    break;


  case STATE_READ_PRESS:
    if ( (Wire.status() != WIRE_STATUS_SUCCESS) || (Wire.available() != 1) )
    {
      state= STATE_FAIL;
    }
    else
    {
      if (meas_data_reg == BMP_DATA_XLSB)
      {
        pval <<= 4;
        pval |= ((unsigned long)Wire.receive() >> 4);
        pressure += pval;
        // request a temperature measurement
        Wire.beginTransmissionAt(BMP_ADDR, BMP_MEAS_RQST);
        Wire.send(BMP_MEAS_TEMP);
        rc = Wire.kickoffTransmission();
        state = (rc == WIRE_STATUS_SUCCESS) ? STATE_MEAS_TEMP : STATE_FAIL;
      }
      else
      {
        pval <<= 8;
        pval |= (unsigned long)Wire.receive();
        rc = Wire.kickoffRequestFromAt(BMP_ADDR, ++meas_data_reg, 1U);
        if (rc != WIRE_STATUS_SUCCESS) state = STATE_FAIL;
      }
    }
    break;

  case STATE_WAIT_NEXT:
    if ( MILLIS_CMP( millis(),  next_reading ) == 1 )
    {
      next_reading += avg_interval;
      Wire.beginTransmissionAt(BMP_ADDR, BMP_MEAS_RQST);
      Wire.send(oss_rqst);
      rc = Wire.kickoffTransmission();
      state = STATE_MEAS_PRESS;
    }
    break;

  case STATE_FAIL: 
    // stuck here with no way out 
    // if the reset pin is connected on the sensor, 
    // we could reset it here...
    if (!failmsg) Serial.println("BMP Failed");
    failmsg=1U;
    break;

  default:
    state = STATE_IDLE; // try to recover
    break;
  }
}
