//============================================================================
//Weather Station Data Logger : Weather Shield for Arduino
//Copyright  2010, Weber Anderson
// 
//This application is free software; you can redistribute it and/or
//modify it under the terms of the GNU Lesser General Public
//License as published by the Free Software Foundation; either
//version 3 of the License, or (at your option) any later version.
//
//This application is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR When PARTICULAR PURPOSE. See the GNU
//Lesser General Public License for more details.
//
//You should have received a copy of the GNU Lesser General Public
//License along with this library; if not, see <http://www.gnu.org/licenses/>
//
//=============================================================================

#include "bmp180Utility.h"
#include "Bmp180.h"
// this uses a modified version of the wire library, which allows
// multiple commands without deselecting the slave.
#include "ExtendedWire.h"

//
// MILLIS_CMP(a,b) returns the sign of (a-b) -- or zero if a and b are identical
// in essence, this does a signed comparison of unsigned long numbers and makes the assumption
// that when two numbers differ by more than mm_diff, that an overflow or underflow must have 
// occurred. the over/underflow is "fixed" and the proper answer is returned
//
#define MILLIS_CMP(a,b) ( (a==b) ? 0 : ( (a>b) ? (((a-b)>mm_diff) ? -1 : 1) : (((b-a)>mm_diff) ? 1 : -1) ) )
const unsigned long mm_diff = 0x7fffffffUL;

// create the instance for users to access the class through.

BoschBmp180 Bmp180 = BoschBmp180();

BoschBmp180::BoschBmp180()
{
	cal_data_available = false;
	data_available = false;
	state = STATE_INIT;
	navg = 1U;
	avg_count = 0;
	meas_rqst = false;
	read_failures = 0;
	//clock_rate = 400000UL;
	oss_mode = 3;
	oss_rqst = BMP_OSS_MODE(oss_mode);
	cal_data = (int*)malloc(sizeof(int) * CAL_DATA_LENGTH);
	twi_initialized = 0;
	failmsg = 0;
	prev_stat = 0xffU;
}

BoschBmp180::~BoschBmp180()
{
	if (cal_data != 0) free(cal_data);
}

void BoschBmp180::init()
{
	if (twi_initialized) return;
	Wire.enablePullupResistors(1);
	Wire.begin(false);
	Wire.setSpeed(100000UL);
	twi_initialized = 1;
}

byte BoschBmp180::initialized()
{
	return twi_initialized;
}

void BoschBmp180::start()
{
	state = STATE_INIT;
	// request the chip id code
	Wire.kickoffRequestFromAt( BMP_ADDR, BMP_CHIP_ID, 1U );
}

void BoschBmp180::waitForPause()
{
	while (state != STATE_IDLE && state != STATE_WAIT_NEXT) step();
}

void BoschBmp180::setupAveraging(byte OSSmode, unsigned int avgCount, unsigned long interval)
{
	if (OSSmode > 3)
		oss_mode = 3;
	else
		oss_mode = OSSmode;

	oss_rqst = BMP_OSS_MODE(oss_mode); // pre-compute the register value to save time

	navg = avgCount;
	avg_interval = interval;
}

boolean BoschBmp180::calDataAvailable()
{
	return cal_data_available;
}

byte BoschBmp180::getCalData(int* buffer, byte length)
{
	if (!cal_data_available) return 0;
	byte count = CAL_DATA_LENGTH;
	if (length < count) return 0;
	memcpy(buffer, cal_data, sizeof(int) * count);
	return count;
}

void BoschBmp180::startMeasurement()
{
	avg_count = 0U;
	temperature = 0UL;
	pressure = 0UL;
	data_available = false;
	meas_rqst = true;
}

boolean BoschBmp180::dataAvailable()
{
	return data_available;
}

boolean BoschBmp180::getRawData(unsigned long* temp, unsigned long* press, unsigned int* avgCount)
{
	if (!data_available) return false;
	*temp = temperature;
	*press = pressure;
	*avgCount = avg_count; // return actual count, not the setting (navg)
	return true;
}


//
// the BMP180 does not have an EOC pin so we must poll the status register to determine 
// if a measurement is completed
//
void BoschBmp180::step()
{
	uint8_t stat = Wire.status();
	uint8_t sreg;

	if (stat != prev_stat)
	{
		//Serial.print("Status: "); Serial.println((int)Wire.status());
		prev_stat = stat;
	}

	if (stat == WIRE_STATUS_BUSY) return;

	uint8_t rc;
	uint8_t cal_offset;
	uint8_t cal_index;

	switch (state)
	{
	case STATE_INIT:
		// we're expecting a read from the chip id register here...
		if ( (Wire.status() != WIRE_STATUS_SUCCESS) || (Wire.available() != 1) )
		{
			// usually caused by the chip being dead -- most likely, it did 
			// not ACK it's address.
			//Serial.print("Fail, avail="); Serial.println((int)Wire.available());
			state = STATE_FAIL;
		}
		else
		{			
			// get the chip id
			//Serial.print("BMP Chip ID is ");
			//Serial.println(Wire.receive());

			if (Wire.receive() != BMP_ID_CODE)
			{
				state = STATE_FAIL;
			}
			else
			{
				// good to go...start reading the EEPROM calibration data
				memset(cal_data, 0, CAL_DATA_LENGTH * sizeof(int));
				cal_data_reg = BMP_CAL_START;
				rc = Wire.kickoffRequestFromAt(BMP_ADDR, cal_data_reg, 1U);
				state = (rc == WIRE_STATUS_SUCCESS) ? STATE_READ_CAL : STATE_FAIL;
				//Serial.println("Read Cal");
			}
		}
		break;

	case STATE_READ_CAL:
		// we're expecting a read from the chip id register here...
		if ( (Wire.status() != WIRE_STATUS_SUCCESS) || (Wire.available() != 1) )
		{
			state = STATE_FAIL;
		}
		else
		{
			cal_offset = cal_data_reg++ - BMP_CAL_START;
			cal_index = cal_offset >> 1;
			cal_data[cal_index] <<= 8;
			cal_data[cal_index] |=  Wire.receive();

			if (cal_data_reg > BMP_CAL_END)
			{
				cal_data_available = 1;
				state = STATE_IDLE;
				//Serial.println("Idle");
			}
			else
			{
				rc = Wire.kickoffRequestFromAt(BMP_ADDR, cal_data_reg, 1U);
				state = (rc == WIRE_STATUS_SUCCESS) ? STATE_READ_CAL : STATE_FAIL;
			}
		}
		break;

	case STATE_IDLE:
		if (!meas_rqst) break; // wait for background loop to request a new msmt
		//
		// start off by requesting a temperature measurement
		//
		Wire.beginTransmissionAt(BMP_ADDR, BMP_MEAS_RQST);
		Wire.send(oss_rqst);
		rc = Wire.kickoffTransmission();
		next_reading = millis() + avg_interval;
		delay_timeout = millis() + 25;				// it will take roughly this long for the reading to complete
		state = (rc == WIRE_STATUS_SUCCESS) ? STATE_QUERY_PRESS : STATE_FAIL;
		//Serial.println("Meas temp");
		break;

	case STATE_QUERY_TEMP:
		if ( millis() >= delay_timeout )
		{			
			Wire.kickoffRequestFromAt(BMP_ADDR, BMP_MEAS_RQST, 1U); // query 1 byte from the control/status register
			state = STATE_CHECK_TEMP;
		}
		break;

	case STATE_CHECK_TEMP:
		if ( (Wire.status() != WIRE_STATUS_SUCCESS) || (Wire.available() != 1))
		{
			state = STATE_FAIL;
		}
		else
		{
			status = Wire.receive();
			if ((status & 0x20) == 0)
			{			
				tval = 0;
				meas_data_reg = BMP_DATA_MSB;
				rc = Wire.kickoffRequestFromAt(BMP_ADDR, meas_data_reg,  1U);
				state = (rc == WIRE_STATUS_SUCCESS) ? STATE_READ_TEMP : STATE_FAIL;
			}
			else
			{
				rc = Wire.kickoffRequestFromAt(BMP_ADDR, BMP_MEAS_RQST, 1U);
				if (rc != WIRE_STATUS_SUCCESS) state = STATE_FAIL;
			}
		}
		break;

	case STATE_READ_TEMP:
		if ( (Wire.status() != WIRE_STATUS_SUCCESS) || (Wire.available() != 1) )
		{
			state = STATE_FAIL;
		}
		else
		{
			tval <<= 8;
			tval |= (unsigned long)Wire.receive();

			if (meas_data_reg == BMP_DATA_LSB)
			{
				temperature += tval;

				if (++avg_count >= navg)
				{
					data_available = true;
					meas_rqst = false;
					state = STATE_IDLE;
				}
				else
				{
					state = STATE_WAIT_NEXT;
				}

			}
			else
			{
				rc = Wire.kickoffRequestFromAt(BMP_ADDR, ++meas_data_reg, 1U);
				if (rc != WIRE_STATUS_SUCCESS) state = STATE_FAIL;
			}
		}
		break;

	case STATE_QUERY_PRESS:
		//
		// previous state should have started a pressure measurement and set the delay timeout for
		// approximate time when the results should be ready. once that timer expires, we kick off
		// a read of the status register and move on to the next state which will read status register 
		// results.
		//
		if (millis() >= delay_timeout)
		{
			// initiate a query of the status register to see if the msmt is completed yet
			rc = Wire.kickoffRequestFromAt(BMP_ADDR, BMP_MEAS_RQST, 1U);
			state = STATE_CHECK_PRESS;
		}
		break;

	case STATE_CHECK_PRESS:
		//
		// we expect that the previous state initiated a read of the status register
		// and this code will not execute until that read has completed
		//
		if ( (Wire.status() != WIRE_STATUS_SUCCESS) || (Wire.available() != 1) )
		{
			state = STATE_FAIL;
		}
		else
		{
			status = Wire.receive();
			if ((status & 0x20) == 0) // is the msmt done?
			{
				// Kick off reading of the pressure data
				pval = 0;
				meas_data_reg = BMP_DATA_MSB;
				// temperature measurement is finished. request the data.
				rc = Wire.kickoffRequestFromAt(BMP_ADDR, meas_data_reg,  1U);
				state = (rc == WIRE_STATUS_SUCCESS) ? STATE_READ_PRESS : STATE_FAIL;
			}
			else
			{
				// request another status byte and just keep looping in this state
				rc = Wire.kickoffRequestFromAt(BMP_ADDR, BMP_MEAS_RQST, 1U);
				if (rc != WIRE_STATUS_SUCCESS) state = STATE_FAIL;
			}
		}
		break;


	case STATE_READ_PRESS:
		if ( (Wire.status() != WIRE_STATUS_SUCCESS) || (Wire.available() != 1) )
		{
			state= STATE_FAIL;
		}
		else
		{
			if (meas_data_reg == BMP_DATA_XLSB)
			{
				pval <<= 4;
				pval |= ((unsigned long)Wire.receive() >> 4);
				pressure += pval;
				// request a temperature measurement
				Wire.beginTransmissionAt(BMP_ADDR, BMP_MEAS_RQST);
				Wire.send(BMP_MEAS_TEMP);
				rc = Wire.kickoffTransmission();
				delay_timeout = millis() + 4;
				state = (rc == WIRE_STATUS_SUCCESS) ? STATE_QUERY_TEMP : STATE_FAIL;
			}
			else
			{
				pval <<= 8;
				pval |= (unsigned long)Wire.receive();
				rc = Wire.kickoffRequestFromAt(BMP_ADDR, ++meas_data_reg, 1U);
				if (rc != WIRE_STATUS_SUCCESS) state = STATE_FAIL;
			}
		}
		break;

	case STATE_WAIT_NEXT:
		if ( MILLIS_CMP( millis(),  next_reading ) == 1 )
		{
			next_reading += avg_interval;
			Wire.beginTransmissionAt(BMP_ADDR, BMP_MEAS_RQST);
			Wire.send(oss_rqst);
			rc = Wire.kickoffTransmission();
			state = STATE_MEAS_PRESS;
		}
		break;

	case STATE_FAIL: 
		// stuck here with no way out 
		// if the reset pin is connected on the sensor, 
		// we could reset it here...
		if (!failmsg) Serial.println("BMP Failed");
		failmsg=1U;
		break;

	default:
		state = STATE_IDLE; // try to recover
		break;
	}
}
