//============================================================================
//Weather Station Data Logger : Weather Shield for Arduino
//Copyright  2010, Weber Anderson
// 
//This application is free software; you can redistribute it and/or
//modify it under the terms of the GNU Lesser General Public
//License as published by the Free Software Foundation; either
//version 3 of the License, or (at your option) any later version.
//
//This application is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR When PARTICULAR PURPOSE. See the GNU
//Lesser General Public License for more details.
//
//You should have received a copy of the GNU Lesser General Public
//License along with this library; if not, see <http://www.gnu.org/licenses/>
//
//=============================================================================
//
// Hardware connections and notes:
//
// The SHT1x sensor only has two signals, clock and data. The clock goes to 
// Arduino pin 3 and the data pin goes to pin 4. It is a good idea to include a 
// 1K series resistor on both pins. If there is any problem with data integrity,
// a 20k pull-up resistor can be attached to the data pin. The SHT1x sensors 
// run on the 5V supply, so level translation is not required. For future improvment,
// if a 10k pull-up resistor is installed on the data line, then driving the line
// from Arduino is simpler -- the data port bit can be set to zero and left there.
// then just changing the port from output to input mode is all that is needed to
// send a zero or one, or read an input bit.
//

#ifndef Sensirion_h
#define Sensirion_h

#define ENABLE_CORRECTED_DATA true

#include "Arduino.h"

class Sht1xSensors
{
public:
  Sht1xSensors();
  void init();
  void init(byte clock_pin, byte data_pin);
  void recover();
  void step();
  boolean dataReady();
  boolean dataValid();
  void start_measurement();
#if ENABLE_CORRECTED_DATA
  boolean getCorrectedData(float *temperatureDegC, float *rhPct);
#endif
  boolean getRawData(unsigned int *tempBits, unsigned int *rhBits);

private:
  void interface_reset();
  boolean send_cmd(byte cmd);
  byte read_byte(boolean msb_first, boolean ack);
  void read_data(unsigned int *value, byte *crc);
  boolean verify_crc8(byte cmd, unsigned int value, byte crc);
  float rdg2temp(unsigned int rdg);
  float rdg2rh(unsigned int rdg, float temp);

  byte sht1x_clock_pin;
  byte sht1x_data_pin;
  byte state;
  unsigned long delay_timeout;
  boolean data_ready;
  boolean msmt_rqst;
  boolean crc_failed;
  boolean ack_failed;
  int bad_crc_count;

  unsigned int rawTemp;
  unsigned int rawRh;

#if ENABLE_CORRECTED_DATA
  float tempDegC;
  float rhPct;
#endif

};

extern Sht1xSensors Sensirion;


#endif
